/*
  Definition of constant tables
*/

#include <stdio.h>
#include "aac_rom.h"
#include "FloatFR.h"


/* 49 scfbands */
static const short sfb_48_1024[] = {
     0,   4,   8,  12,  16,  20,  24,
    28,  32,  36,  40,  48,  56,  64,
    72,  80,  88,  96, 108, 120, 132,
   144, 160, 176, 196, 216, 240, 264,
   292, 320, 352, 384, 416, 448, 480,
   512, 544, 576, 608, 640, 672, 704,
   736, 768, 800, 832, 864, 896, 928,
  1024
};

/* 14 scfbands */
static const short sfb_48_128[] =
{
     0,   4,   8,  12,  16,  20,  28,
    36,  44,  56,  68,  80,  96, 112,
   128
};

/* 51 scfbands */
static const short sfb_32_1024[] =
{
     0,   4,   8,  12,  16,  20,  24,
    28,  32,  36,  40,  48,  56,  64,
    72,  80,  88,  96, 108, 120, 132,
   144, 160, 176, 196, 216, 240, 264,
   292, 320, 352, 384, 416, 448, 480,
   512, 544, 576, 608, 640, 672, 704,
   736, 768, 800, 832, 864, 896, 928,
   960, 992,1024
};

/* 47 scfbands */
static const short sfb_24_1024[] =
{
     0,   4,   8,  12,  16,  20,  24,
    28,  32,  36,  40,  44,  52,  60,
    68,  76,  84,  92, 100, 108, 116,
   124, 136, 148, 160, 172, 188, 204,
   220, 240, 260, 284, 308, 336, 364,
   396, 432, 468, 508, 552, 600, 652,
   704, 768, 832, 896, 960,1024
};

/* 15 scfbands */
static const short sfb_24_128[] =
{
     0,   4,   8,  12,  16,  20,  24,
    28,  36,  44,  52,  64,  76,  92,
   108, 128
};

/* 43 scfbands */
static const short sfb_16_1024[] =
{
     0,   8,  16,  24,  32,  40,  48,
    56,  64,  72,  80,  88, 100, 112,
   124, 136, 148, 160, 172, 184, 196,
   212, 228, 244, 260, 280, 300, 320,
   344, 368, 396, 424, 456, 492, 532,
   572, 616, 664, 716, 772, 832, 896,
   960,1024
};

/* 15 scfbands */
static const short sfb_16_128[] =
{
     0,   4,   8,  12,  16,  20,  24,
    28,  32,  40,  48,  60,  72,  88,
   108, 128
};

/* 40 scfbands */
static const short sfb_8_1024[] =
{
     0,  12,  24,  36,  48,  60,  72,
    84,  96, 108, 120, 132, 144, 156,
   172, 188, 204, 220, 236, 252, 268,
   288, 308, 328, 348, 372, 396, 420,
   448, 476, 508, 544, 580, 620, 664,
   712, 764, 820, 880, 944,1024
};

/* 15 scfbands */
static const short sfb_8_128[] =
{
     0,   4,   8,  12,  16,  20,  24,
    28,  36,  44,  52,  60,  72,  88,
   108, 128
};

/*
  Table entries are sorted as following:
  | fs | num_swb_long_window | sfbands_long | num_swb_short_window | sfbands_short |
*/
const SamplingRateInfo SamplingRateInfoTable[9] =
{
  { 48000, 49, sfb_48_1024, 14, sfb_48_128 }, /* 48000 */
  { 44100, 49, sfb_48_1024, 14, sfb_48_128 }, /* 44100 */
  { 32000, 51, sfb_32_1024, 14, sfb_48_128 }, /* 32000 */
  { 24000, 47, sfb_24_1024, 15, sfb_24_128 }, /* 24000 */
  { 22050, 47, sfb_24_1024, 15, sfb_24_128 }, /* 22050 */
  { 16000, 43, sfb_16_1024, 15, sfb_16_128 }, /* 16000 */
  { 12000, 43, sfb_16_1024, 15, sfb_16_128 }, /* 12000 */
  { 11025, 43, sfb_16_1024, 15, sfb_16_128 }, /* 11025 */
  {  8000, 40, sfb_8_1024,  15, sfb_8_128  }  /*  8000 */
};


/*
  MPEG-2 AAC 2 BITS parallel Hufman Tables

  Bit 0:     = 1=ENDNODE, 0=INDEX
  Bit 1:     = CODEWORD LEN MOD 2
  Bit 2..9:  = VALUE/REF Tables 1..10,SCL
  Bit 2..11: = VALUE/REF Table 11
*/
static const unsigned short HuffmanCodeBook_1[51][4] =
{
  {0x00000157,0x00000157,0x00000004,0x00000018},
  {0x00000008,0x0000000c,0x00000010,0x00000014},
  {0x0000015b,0x0000015b,0x00000153,0x00000153},
  {0x00000057,0x00000057,0x00000167,0x00000167},
  {0x00000257,0x00000257,0x00000117,0x00000117},
  {0x00000197,0x00000197,0x00000147,0x00000147},
  {0x0000001c,0x00000030,0x00000044,0x00000058},
  {0x00000020,0x00000024,0x00000028,0x0000002c},
  {0x0000014b,0x0000014b,0x00000163,0x00000163},
  {0x00000217,0x00000217,0x00000127,0x00000127},
  {0x00000187,0x00000187,0x00000097,0x00000097},
  {0x0000016b,0x0000016b,0x00000017,0x00000017},
  {0x00000034,0x00000038,0x0000003c,0x00000040},
  {0x00000143,0x00000143,0x00000107,0x00000107},
  {0x0000011b,0x0000011b,0x00000067,0x00000067},
  {0x00000193,0x00000193,0x00000297,0x00000297},
  {0x0000019b,0x0000019b,0x00000247,0x00000247},
  {0x00000048,0x0000004c,0x00000050,0x00000054},
  {0x000001a7,0x000001a7,0x00000267,0x00000267},
  {0x00000113,0x00000113,0x0000025b,0x0000025b},
  {0x00000053,0x00000053,0x0000005b,0x0000005b},
  {0x00000253,0x00000253,0x00000047,0x00000047},
  {0x0000005c,0x00000070,0x00000084,0x00000098},
  {0x00000060,0x00000064,0x00000068,0x0000006c},
  {0x0000012b,0x0000012b,0x00000123,0x00000123},
  {0x0000018b,0x0000018b,0x000000a7,0x000000a7},
  {0x00000227,0x00000227,0x00000287,0x00000287},
  {0x00000087,0x00000087,0x0000010b,0x0000010b},
  {0x00000074,0x00000078,0x0000007c,0x00000080},
  {0x0000021b,0x0000021b,0x00000027,0x00000027},
  {0x000001a3,0x000001a3,0x00000093,0x00000093},
  {0x00000183,0x00000183,0x00000207,0x00000207},
  {0x0000024b,0x0000024b,0x0000004b,0x0000004b},
  {0x00000088,0x0000008c,0x00000090,0x00000094},
  {0x00000063,0x00000063,0x00000103,0x00000103},
  {0x00000007,0x00000007,0x000002a7,0x000002a7},
  {0x0000009b,0x0000009b,0x0000026b,0x0000026b},
  {0x00000263,0x00000263,0x000001ab,0x000001ab},
  {0x0000009c,0x000000a0,0x000000a4,0x000000b8},
  {0x00000241,0x00000011,0x00000069,0x00000019},
  {0x00000211,0x00000041,0x00000291,0x00000299},
  {0x000000a8,0x000000ac,0x000000b0,0x000000b4},
  {0x0000008b,0x0000008b,0x00000223,0x00000223},
  {0x000000a3,0x000000a3,0x0000020b,0x0000020b},
  {0x000002ab,0x000002ab,0x00000283,0x00000283},
  {0x0000002b,0x0000002b,0x00000083,0x00000083},
  {0x000000bc,0x000000c0,0x000000c4,0x000000c8},
  {0x00000003,0x00000003,0x0000022b,0x0000022b},
  {0x0000028b,0x0000028b,0x000002a3,0x000002a3},
  {0x00000023,0x00000023,0x00000203,0x00000203},
  {0x0000000b,0x0000000b,0x000000ab,0x000000ab}
};

static const unsigned short HuffmanCodeBook_2[39][4] =
{
  {0x00000004,0x0000000c,0x00000020,0x00000034},
  {0x00000157,0x00000157,0x00000159,0x00000008},
  {0x00000153,0x00000153,0x00000257,0x00000257},
  {0x00000010,0x00000014,0x00000018,0x0000001c},
  {0x00000117,0x00000117,0x00000057,0x00000057},
  {0x00000147,0x00000147,0x00000197,0x00000197},
  {0x00000167,0x00000167,0x00000185,0x00000161},
  {0x00000125,0x00000095,0x00000065,0x00000215},
  {0x00000024,0x00000028,0x0000002c,0x00000030},
  {0x00000051,0x00000149,0x00000119,0x00000141},
  {0x00000015,0x00000199,0x00000259,0x00000245},
  {0x00000191,0x00000265,0x00000105,0x00000251},
  {0x00000045,0x00000111,0x00000169,0x000001a5},
  {0x00000038,0x00000044,0x00000058,0x0000006c},
  {0x00000295,0x00000059,0x0000003c,0x00000040},
  {0x00000227,0x00000227,0x0000021b,0x0000021b},
  {0x00000123,0x00000123,0x00000087,0x00000087},
  {0x00000048,0x0000004c,0x00000050,0x00000054},
  {0x0000018b,0x0000018b,0x0000006b,0x0000006b},
  {0x0000029b,0x0000029b,0x000001a3,0x000001a3},
  {0x00000207,0x00000207,0x000001ab,0x000001ab},
  {0x00000093,0x00000093,0x00000103,0x00000103},
  {0x0000005c,0x00000060,0x00000064,0x00000068},
  {0x00000213,0x00000213,0x0000010b,0x0000010b},
  {0x0000012b,0x0000012b,0x00000249,0x00000061},
  {0x00000181,0x00000291,0x00000241,0x00000041},
  {0x00000005,0x00000099,0x00000019,0x00000025},
  {0x00000070,0x00000074,0x00000078,0x00000088},
  {0x000002a5,0x00000261,0x00000011,0x000000a5},
  {0x00000049,0x00000285,0x00000269,0x00000089},
  {0x00000221,0x0000007c,0x00000080,0x00000084},
  {0x0000020b,0x0000020b,0x00000003,0x00000003},
  {0x000000a3,0x000000a3,0x000002a3,0x000002a3},
  {0x000002ab,0x000002ab,0x00000083,0x00000083},
  {0x0000008c,0x00000090,0x00000094,0x00000098},
  {0x0000028b,0x0000028b,0x00000023,0x00000023},
  {0x00000283,0x00000283,0x0000002b,0x0000002b},
  {0x0000000b,0x0000000b,0x00000203,0x00000203},
  {0x0000022b,0x0000022b,0x000000ab,0x000000ab}
};

static const unsigned short HuffmanCodeBook_3[39][4] =
{
  {0x00000003,0x00000003,0x00000004,0x00000008},
  {0x00000005,0x00000101,0x00000011,0x00000041},
  {0x0000000c,0x00000010,0x00000014,0x00000020},
  {0x00000017,0x00000017,0x00000143,0x00000143},
  {0x00000051,0x00000111,0x00000045,0x00000151},
  {0x00000105,0x00000055,0x00000018,0x0000001c},
  {0x00000157,0x00000157,0x00000147,0x00000147},
  {0x00000117,0x00000117,0x00000009,0x00000201},
  {0x00000024,0x0000002c,0x00000040,0x00000054},
  {0x00000241,0x00000019,0x00000065,0x00000028},
  {0x00000183,0x00000183,0x00000193,0x00000193},
  {0x00000030,0x00000034,0x00000038,0x0000003c},
  {0x00000027,0x00000027,0x00000253,0x00000253},
  {0x0000005b,0x0000005b,0x00000083,0x00000083},
  {0x00000063,0x00000063,0x00000093,0x00000093},
  {0x00000023,0x00000023,0x00000213,0x00000213},
  {0x00000044,0x00000048,0x0000004c,0x00000050},
  {0x0000004b,0x0000004b,0x00000167,0x00000167},
  {0x00000163,0x00000163,0x00000097,0x00000097},
  {0x00000197,0x00000197,0x00000125,0x00000085},
  {0x00000185,0x00000121,0x00000159,0x00000255},
  {0x00000058,0x0000005c,0x00000060,0x00000070},
  {0x00000119,0x00000245,0x00000281,0x00000291},
  {0x00000069,0x000000a5,0x00000205,0x00000109},
  {0x000001a1,0x00000064,0x00000068,0x0000006c},
  {0x0000002b,0x0000002b,0x000001a7,0x000001a7},
  {0x00000217,0x00000217,0x0000014b,0x0000014b},
  {0x00000297,0x00000297,0x0000016b,0x0000016b},
  {0x00000074,0x00000078,0x0000007c,0x00000080},
  {0x000000a3,0x000000a3,0x00000263,0x00000263},
  {0x00000285,0x00000129,0x00000099,0x000000a9},
  {0x000002a1,0x000001a9,0x00000199,0x00000265},
  {0x000002a5,0x00000084,0x00000088,0x0000008c},
  {0x00000223,0x00000223,0x0000008b,0x0000008b},
  {0x00000227,0x00000227,0x00000189,0x00000259},
  {0x00000219,0x00000090,0x00000094,0x00000098},
  {0x000002ab,0x000002ab,0x0000026b,0x0000026b},
  {0x0000029b,0x0000029b,0x0000024b,0x0000024b},
  {0x0000020b,0x0000020b,0x00000229,0x00000289}
};

static const unsigned short HuffmanCodeBook_4[38][4] =
{
  {0x00000004,0x00000008,0x0000000c,0x00000018},
  {0x00000155,0x00000151,0x00000115,0x00000055},
  {0x00000145,0x00000005,0x00000015,0x00000001},
  {0x00000141,0x00000045,0x00000010,0x00000014},
  {0x00000107,0x00000107,0x00000053,0x00000053},
  {0x00000103,0x00000103,0x00000113,0x00000113},
  {0x0000001c,0x00000020,0x00000034,0x00000048},
  {0x00000043,0x00000043,0x00000013,0x00000013},
  {0x00000024,0x00000028,0x0000002c,0x00000030},
  {0x0000015b,0x0000015b,0x00000197,0x00000197},
  {0x00000167,0x00000167,0x00000257,0x00000257},
  {0x0000005b,0x0000005b,0x0000011b,0x0000011b},
  {0x00000067,0x00000067,0x0000014b,0x0000014b},
  {0x00000038,0x0000003c,0x00000040,0x00000044},
  {0x00000193,0x00000193,0x00000251,0x00000095},
  {0x00000161,0x00000245,0x00000125,0x00000215},
  {0x00000185,0x00000019,0x00000049,0x00000025},
  {0x00000109,0x00000211,0x00000061,0x00000241},
  {0x0000004c,0x00000050,0x00000058,0x0000006c},
  {0x00000091,0x00000121,0x00000205,0x00000181},
  {0x00000085,0x00000009,0x00000201,0x00000054},
  {0x00000023,0x00000023,0x00000083,0x00000083},
  {0x0000005c,0x00000060,0x00000064,0x00000068},
  {0x000001a7,0x000001a7,0x0000016b,0x0000016b},
  {0x0000019b,0x0000019b,0x00000297,0x00000297},
  {0x00000267,0x00000267,0x0000025b,0x0000025b},
  {0x000000a5,0x00000069,0x00000099,0x000001a1},
  {0x00000070,0x00000074,0x00000078,0x00000084},
  {0x00000291,0x00000129,0x00000261,0x00000189},
  {0x00000285,0x000001a9,0x00000225,0x00000249},
  {0x00000219,0x000002a5,0x0000007c,0x00000080},
  {0x0000029b,0x0000029b,0x0000026b,0x0000026b},
  {0x000000a3,0x000000a3,0x0000002b,0x0000002b},
  {0x00000088,0x0000008c,0x00000090,0x00000094},
  {0x00000283,0x00000283,0x0000008b,0x0000008b},
  {0x00000223,0x00000223,0x0000020b,0x0000020b},
  {0x000002ab,0x000002ab,0x000002a3,0x000002a3},
  {0x000000ab,0x000000ab,0x00000229,0x00000289}
};

static const unsigned short HuffmanCodeBook_5[41][4] =
{
  {0x00000113,0x00000113,0x00000004,0x00000008},
  {0x0000010d,0x00000115,0x00000151,0x000000d1},
  {0x0000000c,0x00000010,0x00000014,0x00000028},
  {0x000000d7,0x000000d7,0x0000014f,0x0000014f},
  {0x000000cf,0x000000cf,0x00000157,0x00000157},
  {0x00000018,0x0000001c,0x00000020,0x00000024},
  {0x0000010b,0x0000010b,0x00000193,0x00000193},
  {0x0000011b,0x0000011b,0x00000093,0x00000093},
  {0x000000c9,0x00000159,0x0000008d,0x00000195},
  {0x00000149,0x000000d9,0x0000018d,0x00000095},
  {0x0000002c,0x00000030,0x00000044,0x00000058},
  {0x00000105,0x0000011d,0x00000051,0x000001d1},
  {0x00000034,0x00000038,0x0000003c,0x00000040},
  {0x000000c7,0x000000c7,0x000001d7,0x000001d7},
  {0x0000015f,0x0000015f,0x0000004f,0x0000004f},
  {0x00000147,0x00000147,0x000000df,0x000000df},
  {0x00000057,0x00000057,0x000001cf,0x000001cf},
  {0x00000048,0x0000004c,0x00000050,0x00000054},
  {0x0000018b,0x0000018b,0x0000019b,0x0000019b},
  {0x0000008b,0x0000008b,0x0000009b,0x0000009b},
  {0x00000085,0x0000009d,0x000001c9,0x00000059},
  {0x0000019d,0x000001d9,0x00000185,0x00000049},
  {0x0000005c,0x00000060,0x00000074,0x00000088},
  {0x00000011,0x00000101,0x00000161,0x00000121},
  {0x00000064,0x00000068,0x0000006c,0x00000070},
  {0x000000c3,0x000000c3,0x00000213,0x00000213},
  {0x000000e3,0x000000e3,0x0000000f,0x0000000f},
  {0x00000217,0x00000217,0x0000020f,0x0000020f},
  {0x00000143,0x00000143,0x00000017,0x00000017},
  {0x00000078,0x0000007c,0x00000080,0x00000084},
  {0x0000005f,0x0000005f,0x00000047,0x00000047},
  {0x000001c7,0x000001c7,0x0000020b,0x0000020b},
  {0x00000083,0x00000083,0x000001a3,0x000001a3},
  {0x0000001b,0x0000001b,0x0000021b,0x0000021b},
  {0x0000008c,0x00000090,0x00000094,0x00000098},
  {0x000001df,0x000001df,0x00000183,0x00000183},
  {0x00000009,0x000000a1,0x0000001d,0x00000041},
  {0x000001c1,0x0000021d,0x00000205,0x000001e1},
  {0x00000061,0x00000005,0x0000009c,0x000000a0},
  {0x00000023,0x00000023,0x00000203,0x00000203},
  {0x00000223,0x00000223,0x00000003,0x00000003}
};

static const unsigned short HuffmanCodeBook_6[40][4] =
{
  {0x00000004,0x00000008,0x0000000c,0x0000001c},
  {0x00000111,0x00000115,0x000000d1,0x00000151},
  {0x0000010d,0x00000155,0x0000014d,0x000000d5},
  {0x000000cd,0x00000010,0x00000014,0x00000018},
  {0x000000d9,0x00000159,0x00000149,0x000000c9},
  {0x00000109,0x0000018d,0x00000119,0x00000095},
  {0x00000195,0x00000091,0x0000008d,0x00000191},
  {0x00000020,0x00000024,0x00000038,0x0000004c},
  {0x00000099,0x00000189,0x00000089,0x00000199},
  {0x00000028,0x0000002c,0x00000030,0x00000034},
  {0x00000147,0x00000147,0x0000015f,0x0000015f},
  {0x000000df,0x000000df,0x000001cf,0x000001cf},
  {0x000000c7,0x000000c7,0x000001d7,0x000001d7},
  {0x00000057,0x00000057,0x0000004f,0x0000004f},
  {0x0000003c,0x00000040,0x00000044,0x00000048},
  {0x0000011f,0x0000011f,0x00000107,0x00000107},
  {0x00000053,0x00000053,0x000001d3,0x000001d3},
  {0x0000019f,0x0000019f,0x00000085,0x000001c9},
  {0x000001d9,0x0000009d,0x00000059,0x00000049},
  {0x00000050,0x0000005c,0x00000070,0x00000084},
  {0x00000185,0x000001dd,0x00000054,0x00000058},
  {0x0000005f,0x0000005f,0x00000047,0x00000047},
  {0x000001c7,0x000001c7,0x00000017,0x00000017},
  {0x00000060,0x00000064,0x00000068,0x0000006c},
  {0x0000000f,0x0000000f,0x00000163,0x00000163},
  {0x00000143,0x00000143,0x000000c3,0x000000c3},
  {0x00000217,0x00000217,0x000000e3,0x000000e3},
  {0x0000020f,0x0000020f,0x00000013,0x00000013},
  {0x00000074,0x00000078,0x0000007c,0x00000080},
  {0x00000183,0x00000183,0x00000083,0x00000083},
  {0x0000021b,0x0000021b,0x0000000b,0x0000000b},
  {0x00000103,0x00000103,0x000001a3,0x000001a3},
  {0x000000a3,0x000000a3,0x0000020b,0x0000020b},
  {0x00000088,0x0000008c,0x00000090,0x00000094},
  {0x00000123,0x00000123,0x0000001b,0x0000001b},
  {0x00000213,0x00000213,0x00000005,0x00000205},
  {0x0000001d,0x00000061,0x0000021d,0x000001e1},
  {0x000001c1,0x00000041,0x00000098,0x0000009c},
  {0x00000223,0x00000223,0x00000203,0x00000203},
  {0x00000003,0x00000003,0x00000023,0x00000023}
};

static const unsigned short HuffmanCodeBook_7[31][4] =
{
  {0x00000003,0x00000003,0x00000004,0x00000008},
  {0x00000007,0x00000007,0x00000043,0x00000043},
  {0x00000045,0x0000000c,0x00000010,0x00000024},
  {0x00000049,0x00000085,0x00000009,0x00000081},
  {0x00000014,0x00000018,0x0000001c,0x00000020},
  {0x0000004f,0x0000004f,0x000000c7,0x000000c7},
  {0x0000008b,0x0000008b,0x0000000f,0x0000000f},
  {0x000000c3,0x000000c3,0x000000c9,0x0000008d},
  {0x00000105,0x00000051,0x00000145,0x00000055},
  {0x00000028,0x0000002c,0x00000040,0x00000054},
  {0x000000cd,0x00000109,0x00000101,0x00000011},
  {0x00000030,0x00000034,0x00000038,0x0000003c},
  {0x00000093,0x00000093,0x0000014b,0x0000014b},
  {0x00000097,0x00000097,0x00000143,0x00000143},
  {0x0000005b,0x0000005b,0x00000017,0x00000017},
  {0x00000187,0x00000187,0x000000d3,0x000000d3},
  {0x00000044,0x00000048,0x0000004c,0x00000050},
  {0x0000014f,0x0000014f,0x0000010f,0x0000010f},
  {0x000000d7,0x000000d7,0x0000018b,0x0000018b},
  {0x0000009b,0x0000009b,0x000001c7,0x000001c7},
  {0x0000018d,0x00000181,0x00000019,0x00000111},
  {0x00000058,0x0000005c,0x00000060,0x00000068},
  {0x0000005d,0x00000151,0x0000009d,0x00000115},
  {0x000000d9,0x000001c9,0x000000dd,0x00000119},
  {0x00000155,0x00000191,0x000001cd,0x00000064},
  {0x0000001f,0x0000001f,0x000001c3,0x000001c3},
  {0x0000006c,0x00000070,0x00000074,0x00000078},
  {0x0000015b,0x0000015b,0x00000197,0x00000197},
  {0x0000011f,0x0000011f,0x000001d3,0x000001d3},
  {0x000001d7,0x000001d7,0x0000015f,0x0000015f},
  {0x0000019d,0x00000199,0x000001d9,0x000001dd}
};

static const unsigned short HuffmanCodeBook_8[31][4] =
{
  {0x00000004,0x00000008,0x00000010,0x00000024},
  {0x00000047,0x00000047,0x00000049,0x00000005},
  {0x00000085,0x00000041,0x00000089,0x0000000c},
  {0x00000003,0x00000003,0x0000000b,0x0000000b},
  {0x00000014,0x00000018,0x0000001c,0x00000020},
  {0x00000083,0x00000083,0x0000004f,0x0000004f},
  {0x000000c7,0x000000c7,0x0000008f,0x0000008f},
  {0x000000cb,0x000000cb,0x000000cd,0x00000051},
  {0x00000105,0x00000091,0x00000109,0x0000000d},
  {0x00000028,0x0000002c,0x00000040,0x00000054},
  {0x000000c1,0x000000d1,0x0000010d,0x00000095},
  {0x00000030,0x00000034,0x00000038,0x0000003c},
  {0x00000057,0x00000057,0x0000014b,0x0000014b},
  {0x00000147,0x00000147,0x000000d7,0x000000d7},
  {0x0000014f,0x0000014f,0x00000113,0x00000113},
  {0x00000117,0x00000117,0x00000103,0x00000103},
  {0x00000044,0x00000048,0x0000004c,0x00000050},
  {0x00000153,0x00000153,0x00000013,0x00000013},
  {0x0000018b,0x0000018b,0x0000009b,0x0000009b},
  {0x0000005b,0x0000005b,0x00000187,0x00000187},
  {0x0000018d,0x000000d9,0x00000155,0x00000015},
  {0x00000058,0x0000005c,0x00000060,0x00000068},
  {0x00000119,0x00000141,0x00000191,0x0000005d},
  {0x0000009d,0x000001c9,0x00000159,0x000000dd},
  {0x000001c5,0x00000195,0x000001cd,0x00000064},
  {0x0000019b,0x0000019b,0x0000011f,0x0000011f},
  {0x0000006c,0x00000070,0x00000074,0x00000078},
  {0x0000001b,0x0000001b,0x000001d3,0x000001d3},
  {0x00000183,0x00000183,0x0000015f,0x0000015f},
  {0x0000019f,0x0000019f,0x000001db,0x000001db},
  {0x000001d5,0x0000001d,0x000001c1,0x000001dd}
};

static const unsigned short HuffmanCodeBook_9[84][4] =
{
  {0x00000003,0x00000003,0x00000004,0x00000008},
  {0x00000007,0x00000007,0x00000043,0x00000043},
  {0x00000045,0x0000000c,0x00000010,0x0000002c},
  {0x00000049,0x00000085,0x00000009,0x00000081},
  {0x00000014,0x00000018,0x0000001c,0x00000020},
  {0x0000004f,0x0000004f,0x0000008b,0x0000008b},
  {0x000000c7,0x000000c7,0x0000000d,0x000000c1},
  {0x000000c9,0x0000008d,0x00000105,0x00000051},
  {0x00000109,0x00000145,0x00000024,0x00000028},
  {0x00000093,0x00000093,0x000000cf,0x000000cf},
  {0x00000103,0x00000103,0x00000013,0x00000013},
  {0x00000030,0x00000044,0x00000058,0x000000a4},
  {0x00000034,0x00000038,0x0000003c,0x00000040},
  {0x00000057,0x00000057,0x0000014b,0x0000014b},
  {0x00000187,0x00000187,0x0000010f,0x0000010f},
  {0x00000097,0x00000097,0x0000005b,0x0000005b},
  {0x000000d3,0x000000d3,0x00000141,0x00000189},
  {0x00000048,0x0000004c,0x00000050,0x00000054},
  {0x00000015,0x000001c5,0x0000014d,0x00000205},
  {0x00000061,0x00000111,0x000000d5,0x00000099},
  {0x0000005d,0x00000181,0x000000a1,0x00000209},
  {0x0000018d,0x000001c9,0x00000151,0x00000065},
  {0x0000005c,0x00000068,0x0000007c,0x00000090},
  {0x00000245,0x0000009d,0x00000060,0x00000064},
  {0x0000001b,0x0000001b,0x00000117,0x00000117},
  {0x000000db,0x000000db,0x000000e3,0x000000e3},
  {0x0000006c,0x00000070,0x00000074,0x00000078},
  {0x000001c3,0x000001c3,0x000000a7,0x000000a7},
  {0x0000020f,0x0000020f,0x00000193,0x00000193},
  {0x000001cf,0x000001cf,0x00000203,0x00000203},
  {0x0000006b,0x0000006b,0x0000011b,0x0000011b},
  {0x00000080,0x00000084,0x00000088,0x0000008c},
  {0x0000024b,0x0000024b,0x00000157,0x00000157},
  {0x00000023,0x00000023,0x0000001f,0x0000001f},
  {0x000000df,0x000000df,0x000000ab,0x000000ab},
  {0x000000e7,0x000000e7,0x00000123,0x00000123},
  {0x00000094,0x00000098,0x0000009c,0x000000a0},
  {0x00000287,0x00000287,0x0000011f,0x0000011f},
  {0x0000015b,0x0000015b,0x00000197,0x00000197},
  {0x00000213,0x00000213,0x000001d3,0x000001d3},
  {0x0000024f,0x0000024f,0x0000006f,0x0000006f},
  {0x000000a8,0x000000bc,0x000000d0,0x000000f4},
  {0x000000ac,0x000000b0,0x000000b4,0x000000b8},
  {0x00000217,0x00000217,0x00000027,0x00000027},
  {0x00000163,0x00000163,0x000000e9,0x00000289},
  {0x00000241,0x000000ad,0x00000125,0x00000199},
  {0x00000071,0x00000251,0x000001a1,0x000002c5},
  {0x000000c0,0x000000c4,0x000000c8,0x000000cc},
  {0x00000165,0x00000129,0x000001d5,0x0000015d},
  {0x000002c9,0x00000305,0x000000b1,0x000000ed},
  {0x0000028d,0x00000255,0x000001d9,0x000001e1},
  {0x0000012d,0x00000281,0x0000019d,0x000000f1},
  {0x000000d4,0x000000d8,0x000000dc,0x000000e0},
  {0x00000029,0x00000169,0x00000291,0x00000219},
  {0x00000309,0x000001a5,0x000001e5,0x000002d1},
  {0x0000002d,0x00000259,0x000002cd,0x00000295},
  {0x000000e4,0x000000e8,0x000000ec,0x000000f0},
  {0x00000223,0x00000223,0x0000021f,0x0000021f},
  {0x00000173,0x00000173,0x0000030f,0x0000030f},
  {0x0000016f,0x0000016f,0x000001df,0x000001df},
  {0x00000133,0x00000133,0x000001af,0x000001af},
  {0x000000f8,0x0000010c,0x00000120,0x00000134},
  {0x000000fc,0x00000100,0x00000104,0x00000108},
  {0x000001ab,0x000001ab,0x00000313,0x00000313},
  {0x0000025f,0x0000025f,0x000002d7,0x000002d7},
  {0x000002c3,0x000002c3,0x000001b3,0x000001b3},
  {0x0000029b,0x0000029b,0x00000033,0x00000033},
  {0x00000110,0x00000114,0x00000118,0x0000011c},
  {0x000001eb,0x000001eb,0x00000317,0x00000317},
  {0x0000029f,0x0000029f,0x00000227,0x00000227},
  {0x00000303,0x00000303,0x000001ef,0x000001ef},
  {0x00000263,0x00000263,0x00000267,0x00000267},
  {0x00000124,0x00000128,0x0000012c,0x00000130},
  {0x0000022b,0x0000022b,0x000002df,0x000002df},
  {0x000001f3,0x000001f3,0x000002db,0x000002db},
  {0x000002e3,0x000002e3,0x0000022f,0x0000022f},
  {0x0000031f,0x0000031f,0x0000031b,0x0000031b},
  {0x00000138,0x0000013c,0x00000140,0x00000144},
  {0x000002a1,0x00000269,0x00000321,0x000002a5},
  {0x000002e5,0x00000325,0x000002e9,0x00000271},
  {0x000002a9,0x0000026d,0x00000231,0x000002ad},
  {0x000002b1,0x000002f1,0x00000148,0x0000014c},
  {0x0000032b,0x0000032b,0x000002ef,0x000002ef},
  {0x0000032f,0x0000032f,0x00000333,0x00000333}
};

static const unsigned short HuffmanCodeBook_10[82][4] =
{
  {0x00000004,0x0000000c,0x00000020,0x0000004c},
  {0x00000045,0x00000085,0x00000049,0x00000008},
  {0x0000008b,0x0000008b,0x00000007,0x00000007},
  {0x00000010,0x00000014,0x00000018,0x0000001c},
  {0x00000043,0x00000043,0x000000c7,0x000000c7},
  {0x0000008f,0x0000008f,0x0000004f,0x0000004f},
  {0x000000cb,0x000000cb,0x000000cf,0x000000cf},
  {0x00000009,0x00000081,0x00000109,0x00000091},
  {0x00000024,0x00000028,0x0000002c,0x00000038},
  {0x00000105,0x00000051,0x00000001,0x000000d1},
  {0x0000010d,0x0000000d,0x000000c1,0x00000111},
  {0x00000149,0x00000095,0x00000030,0x00000034},
  {0x00000147,0x00000147,0x00000057,0x00000057},
  {0x000000d7,0x000000d7,0x0000014f,0x0000014f},
  {0x0000003c,0x00000040,0x00000044,0x00000048},
  {0x00000117,0x00000117,0x00000153,0x00000153},
  {0x0000009b,0x0000009b,0x0000018b,0x0000018b},
  {0x000000db,0x000000db,0x00000013,0x00000013},
  {0x0000005b,0x0000005b,0x00000103,0x00000103},
  {0x00000050,0x00000064,0x00000078,0x000000c0},
  {0x00000054,0x00000058,0x0000005c,0x00000060},
  {0x00000187,0x00000187,0x0000018f,0x0000018f},
  {0x00000157,0x00000157,0x0000011b,0x0000011b},
  {0x00000193,0x00000193,0x00000159,0x0000009d},
  {0x000001cd,0x000001c9,0x00000195,0x000000a1},
  {0x00000068,0x0000006c,0x00000070,0x00000074},
  {0x000000dd,0x00000015,0x0000005d,0x00000141},
  {0x00000061,0x000001c5,0x000000e1,0x0000011d},
  {0x000001d1,0x00000209,0x00000199,0x0000015d},
  {0x00000205,0x0000020d,0x00000121,0x00000211},
  {0x0000007c,0x00000084,0x00000098,0x000000ac},
  {0x000001d5,0x00000161,0x00000215,0x00000080},
  {0x0000019f,0x0000019f,0x000001db,0x000001db},
  {0x00000088,0x0000008c,0x00000090,0x00000094},
  {0x000000a7,0x000000a7,0x0000001b,0x0000001b},
  {0x0000021b,0x0000021b,0x000000e7,0x000000e7},
  {0x0000024f,0x0000024f,0x00000067,0x00000067},
  {0x0000024b,0x0000024b,0x00000183,0x00000183},
  {0x0000009c,0x000000a0,0x000000a4,0x000000a8},
  {0x000001a3,0x000001a3,0x00000127,0x00000127},
  {0x00000253,0x00000253,0x000000ab,0x000000ab},
  {0x00000247,0x00000247,0x000001df,0x000001df},
  {0x000001e3,0x000001e3,0x00000167,0x00000167},
  {0x000000b0,0x000000b4,0x000000b8,0x000000bc},
  {0x0000021f,0x0000021f,0x000000eb,0x000000eb},
  {0x00000257,0x00000257,0x0000012b,0x0000012b},
  {0x0000028b,0x0000028b,0x0000006b,0x0000006b},
  {0x0000028f,0x0000028f,0x000001a7,0x000001a7},
  {0x000000c4,0x000000d8,0x000000ec,0x00000100},
  {0x000000c8,0x000000cc,0x000000d0,0x000000d4},
  {0x0000025b,0x0000025b,0x00000023,0x00000023},
  {0x00000293,0x00000293,0x0000001f,0x0000001f},
  {0x000000af,0x000000af,0x0000025d,0x000000ed},
  {0x000001a9,0x00000285,0x0000006d,0x000001e5},
  {0x000000dc,0x000000e0,0x000000e4,0x000000e8},
  {0x000001c1,0x00000221,0x00000169,0x000002cd},
  {0x00000295,0x00000261,0x0000016d,0x00000201},
  {0x0000012d,0x000002c9,0x0000029d,0x00000299},
  {0x000001e9,0x000002d1,0x000002c5,0x000000b1},
  {0x000000f0,0x000000f4,0x000000f8,0x000000fc},
  {0x00000225,0x000000f1,0x000001ad,0x000002d5},
  {0x00000131,0x000001ed,0x00000171,0x0000030d},
  {0x000002d9,0x00000025,0x00000229,0x00000029},
  {0x00000071,0x00000241,0x00000311,0x00000265},
  {0x00000104,0x0000010c,0x00000120,0x00000134},
  {0x000001b1,0x00000309,0x000002a1,0x00000108},
  {0x000002a7,0x000002a7,0x00000307,0x00000307},
  {0x00000110,0x00000114,0x00000118,0x0000011c},
  {0x0000022f,0x0000022f,0x000001f3,0x000001f3},
  {0x000002df,0x000002df,0x00000317,0x00000317},
  {0x0000031b,0x0000031b,0x0000026b,0x0000026b},
  {0x000002e3,0x000002e3,0x00000233,0x00000233},
  {0x00000124,0x00000128,0x0000012c,0x00000130},
  {0x00000283,0x00000283,0x0000031f,0x0000031f},
  {0x0000002f,0x0000002f,0x000002ab,0x000002ab},
  {0x0000026f,0x0000026f,0x000002af,0x000002af},
  {0x000002c3,0x000002c3,0x000002ef,0x000002ef},
  {0x00000138,0x0000013c,0x00000140,0x00000144},
  {0x000002e7,0x000002e7,0x000002eb,0x000002eb},
  {0x00000033,0x00000033,0x00000323,0x00000323},
  {0x00000271,0x00000329,0x00000325,0x0000032d},
  {0x000002f1,0x00000301,0x000002b1,0x00000331}
};

static const unsigned short HuffmanCodeBook_11[136][4] =
{
  {0x00000004,0x00000010,0x00000038,0x0000008c},
  {0x00000001,0x00000085,0x00000008,0x0000000c},
  {0x00000843,0x00000843,0x00000007,0x00000007},
  {0x00000083,0x00000083,0x0000008b,0x0000008b},
  {0x00000014,0x00000018,0x0000001c,0x00000024},
  {0x00000107,0x00000107,0x0000010b,0x0000010b},
  {0x00000185,0x0000008d,0x0000010d,0x00000009},
  {0x00000189,0x00000101,0x0000018d,0x00000020},
  {0x00000093,0x00000093,0x00000207,0x00000207},
  {0x00000028,0x0000002c,0x00000030,0x00000034},
  {0x00000113,0x00000113,0x0000020b,0x0000020b},
  {0x00000193,0x00000193,0x0000020f,0x0000020f},
  {0x0000000f,0x0000000f,0x00000183,0x00000183},
  {0x00000097,0x00000097,0x00000117,0x00000117},
  {0x0000003c,0x00000050,0x00000064,0x00000078},
  {0x00000040,0x00000044,0x00000048,0x0000004c},
  {0x0000028b,0x0000028b,0x00000213,0x00000213},
  {0x00000287,0x00000287,0x00000197,0x00000197},
  {0x0000028f,0x0000028f,0x00000217,0x00000217},
  {0x00000291,0x00000119,0x00000309,0x00000099},
  {0x00000054,0x00000058,0x0000005c,0x00000060},
  {0x00000199,0x0000030d,0x00000305,0x00000811},
  {0x0000080d,0x000002c1,0x000001c1,0x00000241},
  {0x00000219,0x00000341,0x00000011,0x00000311},
  {0x00000201,0x00000809,0x00000295,0x00000815},
  {0x00000068,0x0000006c,0x00000070,0x00000074},
  {0x000003c1,0x00000141,0x00000441,0x00000389},
  {0x0000011d,0x0000038d,0x00000299,0x00000315},
  {0x00000819,0x00000541,0x0000019d,0x0000009d},
  {0x000004c1,0x0000081d,0x00000805,0x00000385},
  {0x0000007c,0x00000080,0x00000084,0x00000088},
  {0x00000391,0x000005c1,0x0000021d,0x00000641},
  {0x00000821,0x000000c1,0x00000319,0x00000825},
  {0x00000409,0x00000395,0x00000829,0x000006c1},
  {0x000001a1,0x00000121,0x0000040d,0x00000015},
  {0x00000090,0x000000c8,0x0000011c,0x00000170},
  {0x00000094,0x00000098,0x000000a0,0x000000b4},
  {0x00000741,0x0000082d,0x0000029d,0x00000411},
  {0x00000399,0x0000031d,0x00000281,0x0000009c},
  {0x00000223,0x00000223,0x000007c3,0x000007c3},
  {0x000000a4,0x000000a8,0x000000ac,0x000000b0},
  {0x00000833,0x00000833,0x00000407,0x00000407},
  {0x000000a3,0x000000a3,0x0000083b,0x0000083b},
  {0x00000417,0x00000417,0x00000837,0x00000837},
  {0x0000048f,0x0000048f,0x000002a3,0x000002a3},
  {0x000000b8,0x000000bc,0x000000c0,0x000000c4},
  {0x0000039f,0x0000039f,0x0000048b,0x0000048b},
  {0x00000323,0x00000323,0x00000127,0x00000127},
  {0x000001a7,0x000001a7,0x0000083f,0x0000083f},
  {0x00000493,0x00000493,0x0000041b,0x0000041b},
  {0x000000cc,0x000000e0,0x000000f4,0x00000108},
  {0x000000d0,0x000000d4,0x000000d8,0x000000dc},
  {0x0000001b,0x0000001b,0x00000227,0x00000227},
  {0x00000497,0x00000497,0x000003a3,0x000003a3},
  {0x0000041f,0x0000041f,0x00000487,0x00000487},
  {0x000001ab,0x000001ab,0x00000303,0x00000303},
  {0x000000e4,0x000000e8,0x000000ec,0x000000f0},
  {0x0000012b,0x0000012b,0x000000a7,0x000000a7},
  {0x000002a7,0x000002a7,0x00000513,0x00000513},
  {0x0000050b,0x0000050b,0x00000327,0x00000327},
  {0x0000050f,0x0000050f,0x0000049b,0x0000049b},
  {0x000000f8,0x000000fc,0x00000100,0x00000104},
  {0x0000022b,0x0000022b,0x00000423,0x00000423},
  {0x000002ab,0x000002ab,0x000003a7,0x000003a7},
  {0x000001af,0x000001af,0x00000507,0x00000507},
  {0x0000001f,0x0000001f,0x0000032b,0x0000032b},
  {0x0000010c,0x00000110,0x00000114,0x00000118},
  {0x0000049f,0x0000049f,0x0000058f,0x0000058f},
  {0x00000517,0x00000517,0x000000ab,0x000000ab},
  {0x00000593,0x00000593,0x0000012f,0x0000012f},
  {0x00000137,0x00000137,0x0000051b,0x0000051b},
  {0x00000120,0x00000134,0x00000148,0x0000015c},
  {0x00000124,0x00000128,0x0000012c,0x00000130},
  {0x000001b7,0x000001b7,0x0000058b,0x0000058b},
  {0x00000043,0x00000043,0x00000597,0x00000597},
  {0x000002af,0x000002af,0x0000022d,0x00000425},
  {0x0000051d,0x000004a1,0x00000801,0x00000691},
  {0x00000138,0x0000013c,0x00000140,0x00000144},
  {0x00000381,0x0000068d,0x0000032d,0x000000b5},
  {0x00000235,0x000001b1,0x00000689,0x000002b5},
  {0x00000521,0x00000599,0x00000429,0x000003a9},
  {0x00000139,0x00000231,0x00000585,0x00000611},
  {0x0000014c,0x00000150,0x00000154,0x00000158},
  {0x000000ad,0x0000060d,0x00000685,0x00000131},
  {0x0000059d,0x0000070d,0x00000615,0x00000695},
  {0x00000239,0x00000711,0x000003ad,0x000001b9},
  {0x000002b1,0x00000335,0x00000331,0x00000021},
  {0x00000160,0x00000164,0x00000168,0x0000016c},
  {0x0000042d,0x00000609,0x000004a5,0x000002b9},
  {0x00000699,0x00000529,0x0000013d,0x000005a1},
  {0x00000525,0x00000339,0x000004a9,0x00000715},
  {0x000004ad,0x000000b9,0x00000709,0x00000619},
  {0x00000174,0x00000188,0x0000019c,0x000001cc},
  {0x00000178,0x0000017c,0x00000180,0x00000184},
  {0x00000605,0x00000435,0x00000401,0x000003b5},
  {0x0000061d,0x000003b1,0x0000069d,0x000001bd},
  {0x000000b1,0x00000719,0x00000789,0x000002bd},
  {0x0000023d,0x00000705,0x000005a5,0x00000791},
  {0x0000018c,0x00000190,0x00000194,0x00000198},
  {0x000003b9,0x000006a1,0x000004b5,0x00000621},
  {0x00000795,0x0000078d,0x000005a9,0x0000052d},
  {0x00000431,0x0000033d,0x000003bd,0x00000721},
  {0x000000bd,0x0000071d,0x00000025,0x00000481},
  {0x000001a0,0x000001a4,0x000001a8,0x000001b8},
  {0x000006a5,0x00000625,0x000004b1,0x00000439},
  {0x000006a9,0x000004b9,0x00000531,0x00000799},
  {0x0000079d,0x000001ac,0x000001b0,0x000001b4},
  {0x00000727,0x00000727,0x0000043f,0x0000043f},
  {0x000005af,0x000005af,0x0000072f,0x0000072f},
  {0x00000787,0x00000787,0x0000062b,0x0000062b},
  {0x000001bc,0x000001c0,0x000001c4,0x000001c8},
  {0x0000072b,0x0000072b,0x000005b7,0x000005b7},
  {0x00000537,0x00000537,0x000006af,0x000006af},
  {0x0000062f,0x0000062f,0x000007a3,0x000007a3},
  {0x000005bb,0x000005bb,0x00000637,0x00000637},
  {0x000001d0,0x000001e4,0x000001f8,0x0000020c},
  {0x000001d4,0x000001d8,0x000001dc,0x000001e0},
  {0x000006b3,0x000006b3,0x000004bf,0x000004bf},
  {0x0000053b,0x0000053b,0x0000002b,0x0000002b},
  {0x000005b3,0x000005b3,0x000007a7,0x000007a7},
  {0x00000503,0x00000503,0x00000633,0x00000633},
  {0x000001e8,0x000001ec,0x000001f0,0x000001f4},
  {0x0000002f,0x0000002f,0x00000733,0x00000733},
  {0x000007ab,0x000007ab,0x000006b7,0x000006b7},
  {0x00000683,0x00000683,0x0000063b,0x0000063b},
  {0x0000053f,0x0000053f,0x000005bf,0x000005bf},
  {0x000001fc,0x00000200,0x00000204,0x00000208},
  {0x000007af,0x000007af,0x000006bb,0x000006bb},
  {0x00000037,0x00000037,0x00000583,0x00000583},
  {0x00000737,0x00000737,0x0000063f,0x0000063f},
  {0x000006bf,0x000006bf,0x000007b3,0x000007b3},
  {0x00000210,0x00000214,0x00000218,0x0000021c},
  {0x0000003b,0x0000003b,0x0000073b,0x0000073b},
  {0x000007b7,0x000007b7,0x00000033,0x00000033},
  {0x000007bb,0x000007bb,0x00000701,0x00000601},
  {0x0000073d,0x0000003d,0x00000781,0x000007bd}
};

static const unsigned short HuffmanCodeBook_SCL[65][4] =
{
  {0x000000f3,0x000000f3,0x00000004,0x00000008},
  {0x000000ef,0x000000ef,0x000000f5,0x000000e9},
  {0x000000f9,0x0000000c,0x00000010,0x00000014},
  {0x000000e7,0x000000e7,0x000000ff,0x000000ff},
  {0x000000e1,0x00000101,0x000000dd,0x00000105},
  {0x00000018,0x0000001c,0x00000020,0x00000028},
  {0x0000010b,0x0000010b,0x000000db,0x000000db},
  {0x0000010f,0x0000010f,0x000000d5,0x00000111},
  {0x000000d1,0x00000115,0x000000cd,0x00000024},
  {0x0000011b,0x0000011b,0x000000cb,0x000000cb},
  {0x0000002c,0x00000030,0x00000034,0x00000040},
  {0x000000c7,0x000000c7,0x0000011f,0x0000011f},
  {0x00000121,0x000000c1,0x00000125,0x000000bd},
  {0x00000129,0x000000b9,0x00000038,0x0000003c},
  {0x00000133,0x00000133,0x0000012f,0x0000012f},
  {0x00000137,0x00000137,0x0000013b,0x0000013b},
  {0x00000044,0x00000048,0x0000004c,0x00000058},
  {0x000000b7,0x000000b7,0x000000af,0x000000af},
  {0x000000b1,0x0000013d,0x000000a9,0x000000a5},
  {0x00000141,0x000000a1,0x00000050,0x00000054},
  {0x00000147,0x00000147,0x0000009f,0x0000009f},
  {0x0000014b,0x0000014b,0x0000009b,0x0000009b},
  {0x0000005c,0x00000060,0x00000064,0x00000070},
  {0x0000014f,0x0000014f,0x00000095,0x0000008d},
  {0x00000155,0x00000085,0x00000091,0x00000089},
  {0x00000151,0x00000081,0x00000068,0x0000006c},
  {0x0000015f,0x0000015f,0x00000167,0x00000167},
  {0x0000007b,0x0000007b,0x0000007f,0x0000007f},
  {0x00000074,0x00000078,0x00000080,0x000000b0},
  {0x00000159,0x00000075,0x00000069,0x0000006d},
  {0x00000071,0x00000061,0x00000161,0x0000007c},
  {0x00000067,0x00000067,0x0000005b,0x0000005b},
  {0x00000084,0x00000088,0x0000008c,0x0000009c},
  {0x0000005f,0x0000005f,0x00000169,0x00000055},
  {0x0000004d,0x0000000d,0x00000005,0x00000009},
  {0x00000001,0x00000090,0x00000094,0x00000098},
  {0x0000018b,0x0000018b,0x0000018f,0x0000018f},
  {0x00000193,0x00000193,0x00000197,0x00000197},
  {0x0000019b,0x0000019b,0x000001d7,0x000001d7},
  {0x000000a0,0x000000a4,0x000000a8,0x000000ac},
  {0x00000187,0x00000187,0x0000016f,0x0000016f},
  {0x00000173,0x00000173,0x00000177,0x00000177},
  {0x0000017b,0x0000017b,0x0000017f,0x0000017f},
  {0x00000183,0x00000183,0x000001a3,0x000001a3},
  {0x000000b4,0x000000c8,0x000000dc,0x000000f0},
  {0x000000b8,0x000000bc,0x000000c0,0x000000c4},
  {0x000001bf,0x000001bf,0x000001c3,0x000001c3},
  {0x000001c7,0x000001c7,0x000001cb,0x000001cb},
  {0x000001cf,0x000001cf,0x000001d3,0x000001d3},
  {0x000001bb,0x000001bb,0x000001a7,0x000001a7},
  {0x000000cc,0x000000d0,0x000000d4,0x000000d8},
  {0x000001ab,0x000001ab,0x000001af,0x000001af},
  {0x000001b3,0x000001b3,0x000001b7,0x000001b7},
  {0x000001db,0x000001db,0x0000001b,0x0000001b},
  {0x00000023,0x00000023,0x00000027,0x00000027},
  {0x000000e0,0x000000e4,0x000000e8,0x000000ec},
  {0x0000002b,0x0000002b,0x00000017,0x00000017},
  {0x0000019f,0x0000019f,0x000001e3,0x000001e3},
  {0x000001df,0x000001df,0x00000013,0x00000013},
  {0x0000001f,0x0000001f,0x0000003f,0x0000003f},
  {0x000000f4,0x000000f8,0x000000fc,0x00000100},
  {0x00000043,0x00000043,0x0000004b,0x0000004b},
  {0x00000053,0x00000053,0x00000047,0x00000047},
  {0x0000002f,0x0000002f,0x00000033,0x00000033},
  {0x0000003b,0x0000003b,0x00000037,0x00000037}
};

/*
  interleave codebooks the following way

    9 (84w) |  1 (51w)
   10 (82w) |  2 (39w)
  SCL (65w) |  4 (38w)
    3 (39w) |  5 (41w)
            |  6 (40w)
            |  7 (31w)
            |  8 (31w)
     (270w)     (271w)
*/
const CodeBookDescription HuffmanCodeBooks[13] =
{
  { 0, 0, 0,  0L,               },
  { 4, 2, 1,  HuffmanCodeBook_1 },
  { 4, 2, 1,  HuffmanCodeBook_2 },
  { 4, 2, 0,  HuffmanCodeBook_3 },
  { 4, 2, 0,  HuffmanCodeBook_4 },
  { 2, 4, 4,  HuffmanCodeBook_5 },
  { 2, 4, 4,  HuffmanCodeBook_6 },
  { 2, 4, 0,  HuffmanCodeBook_7 },
  { 2, 4, 0,  HuffmanCodeBook_8 },
  { 2, 4, 0,  HuffmanCodeBook_9 },
  { 2, 4, 0,  HuffmanCodeBook_10},
  { 2, 5, 0,  HuffmanCodeBook_11},
  { 1, 8, 60, HuffmanCodeBook_SCL}
} ;

/*! Long Sine Window for frame length 1024 */
const float OnlyLongWindowSine[FRAME_SIZE] =
{
  0.000767F, 0.002301F, 0.003835F, 0.005369F, 0.006903F, 0.008437F, 0.009971F, 0.011505F, 
  0.013038F, 0.014572F, 0.016106F, 0.017640F, 0.019174F, 0.020707F, 0.022241F, 0.023774F, 
  0.025308F, 0.026841F, 0.028375F, 0.029908F, 0.031441F, 0.032975F, 0.034508F, 0.036041F, 
  0.037574F, 0.039107F, 0.040639F, 0.042172F, 0.043705F, 0.045237F, 0.046769F, 0.048302F, 
  0.049834F, 0.051366F, 0.052898F, 0.054429F, 0.055961F, 0.057493F, 0.059024F, 0.060555F, 
  0.062086F, 0.063617F, 0.065148F, 0.066679F, 0.068209F, 0.069739F, 0.071270F, 0.072800F, 
  0.074329F, 0.075859F, 0.077389F, 0.078918F, 0.080447F, 0.081976F, 0.083505F, 0.085033F, 
  0.086561F, 0.088090F, 0.089617F, 0.091145F, 0.092673F, 0.094200F, 0.095727F, 0.097254F, 
  0.098780F, 0.100307F, 0.101833F, 0.103359F, 0.104884F, 0.106410F, 0.107935F, 0.109460F, 
  0.110984F, 0.112509F, 0.114033F, 0.115557F, 0.117080F, 0.118604F, 0.120127F, 0.121649F, 
  0.123172F, 0.124694F, 0.126216F, 0.127737F, 0.129259F, 0.130780F, 0.132300F, 0.133821F, 
  0.135341F, 0.136860F, 0.138380F, 0.139899F, 0.141418F, 0.142936F, 0.144454F, 0.145972F, 
  0.147489F, 0.149006F, 0.150523F, 0.152039F, 0.153555F, 0.155071F, 0.156586F, 0.158101F, 
  0.159615F, 0.161129F, 0.162643F, 0.164157F, 0.165670F, 0.167182F, 0.168694F, 0.170206F, 
  0.171718F, 0.173229F, 0.174739F, 0.176249F, 0.177759F, 0.179268F, 0.180777F, 0.182286F, 
  0.183794F, 0.185301F, 0.186809F, 0.188315F, 0.189822F, 0.191328F, 0.192833F, 0.194338F, 
  0.195843F, 0.197347F, 0.198850F, 0.200353F, 0.201856F, 0.203358F, 0.204860F, 0.206361F, 
  0.207862F, 0.209362F, 0.210862F, 0.212361F, 0.213860F, 0.215358F, 0.216856F, 0.218353F, 
  0.219850F, 0.221346F, 0.222841F, 0.224337F, 0.225831F, 0.227325F, 0.228819F, 0.230312F, 
  0.231804F, 0.233296F, 0.234788F, 0.236278F, 0.237769F, 0.239258F, 0.240748F, 0.242236F, 
  0.243724F, 0.245212F, 0.246698F, 0.248185F, 0.249670F, 0.251155F, 0.252640F, 0.254124F, 
  0.255607F, 0.257090F, 0.258572F, 0.260054F, 0.261534F, 0.263015F, 0.264494F, 0.265973F, 
  0.267452F, 0.268930F, 0.270407F, 0.271883F, 0.273359F, 0.274834F, 0.276309F, 0.277783F, 
  0.279256F, 0.280729F, 0.282201F, 0.283672F, 0.285143F, 0.286613F, 0.288082F, 0.289551F, 
  0.291019F, 0.292486F, 0.293952F, 0.295418F, 0.296883F, 0.298348F, 0.299812F, 0.301275F, 
  0.302737F, 0.304199F, 0.305660F, 0.307120F, 0.308579F, 0.310038F, 0.311496F, 0.312953F, 
  0.314410F, 0.315866F, 0.317321F, 0.318775F, 0.320229F, 0.321682F, 0.323134F, 0.324585F, 
  0.326035F, 0.327485F, 0.328934F, 0.330382F, 0.331830F, 0.333277F, 0.334723F, 0.336168F, 
  0.337612F, 0.339055F, 0.340498F, 0.341940F, 0.343381F, 0.344821F, 0.346261F, 0.347700F, 
  0.349138F, 0.350575F, 0.352011F, 0.353446F, 0.354881F, 0.356314F, 0.357747F, 0.359179F, 
  0.360611F, 0.362041F, 0.363470F, 0.364899F, 0.366327F, 0.367754F, 0.369180F, 0.370605F, 
  0.372029F, 0.373453F, 0.374875F, 0.376297F, 0.377718F, 0.379138F, 0.380557F, 0.381975F, 
  0.383392F, 0.384808F, 0.386224F, 0.387638F, 0.389052F, 0.390464F, 0.391876F, 0.393287F, 
  0.394697F, 0.396106F, 0.397514F, 0.398921F, 0.400327F, 0.401732F, 0.403137F, 0.404540F, 
  0.405942F, 0.407344F, 0.408744F, 0.410144F, 0.411542F, 0.412940F, 0.414337F, 0.415732F, 
  0.417127F, 0.418520F, 0.419913F, 0.421305F, 0.422696F, 0.424085F, 0.425474F, 0.426862F, 
  0.428248F, 0.429634F, 0.431019F, 0.432402F, 0.433785F, 0.435167F, 0.436547F, 0.437927F, 
  0.439305F, 0.440683F, 0.442059F, 0.443435F, 0.444809F, 0.446183F, 0.447555F, 0.448926F, 
  0.450296F, 0.451665F, 0.453034F, 0.454401F, 0.455766F, 0.457131F, 0.458495F, 0.459858F, 
  0.461219F, 0.462580F, 0.463939F, 0.465298F, 0.466655F, 0.468011F, 0.469366F, 0.470720F, 
  0.472073F, 0.473425F, 0.474775F, 0.476125F, 0.477473F, 0.478821F, 0.480167F, 0.481512F, 
  0.482856F, 0.484198F, 0.485540F, 0.486880F, 0.488220F, 0.489558F, 0.490895F, 0.492231F, 
  0.493565F, 0.494899F, 0.496231F, 0.497563F, 0.498893F, 0.500221F, 0.501549F, 0.502876F, 
  0.504201F, 0.505525F, 0.506848F, 0.508170F, 0.509490F, 0.510810F, 0.512128F, 0.513445F, 
  0.514760F, 0.516075F, 0.517388F, 0.518700F, 0.520011F, 0.521321F, 0.522629F, 0.523937F, 
  0.525243F, 0.526547F, 0.527851F, 0.529153F, 0.530454F, 0.531754F, 0.533052F, 0.534350F, 
  0.535645F, 0.536940F, 0.538234F, 0.539526F, 0.540817F, 0.542106F, 0.543395F, 0.544682F, 
  0.545968F, 0.547252F, 0.548536F, 0.549818F, 0.551098F, 0.552378F, 0.553656F, 0.554932F, 
  0.556208F, 0.557482F, 0.558755F, 0.560026F, 0.561297F, 0.562565F, 0.563833F, 0.565099F, 
  0.566364F, 0.567628F, 0.568890F, 0.570151F, 0.571410F, 0.572669F, 0.573925F, 0.575181F, 
  0.576435F, 0.577688F, 0.578939F, 0.580189F, 0.581438F, 0.582686F, 0.583932F, 0.585176F, 
  0.586419F, 0.587661F, 0.588902F, 0.590141F, 0.591378F, 0.592615F, 0.593850F, 0.595083F, 
  0.596315F, 0.597546F, 0.598775F, 0.600003F, 0.601230F, 0.602455F, 0.603678F, 0.604900F, 
  0.606121F, 0.607341F, 0.608559F, 0.609775F, 0.610990F, 0.612204F, 0.613416F, 0.614627F, 
  0.615836F, 0.617044F, 0.618250F, 0.619455F, 0.620659F, 0.621861F, 0.623061F, 0.624260F, 
  0.625458F, 0.626654F, 0.627849F, 0.629042F, 0.630234F, 0.631424F, 0.632613F, 0.633800F, 
  0.634986F, 0.636170F, 0.637353F, 0.638534F, 0.639714F, 0.640892F, 0.642069F, 0.643245F, 
  0.644418F, 0.645590F, 0.646761F, 0.647930F, 0.649098F, 0.650264F, 0.651429F, 0.652592F, 
  0.653753F, 0.654913F, 0.656072F, 0.657229F, 0.658384F, 0.659538F, 0.660690F, 0.661841F, 
  0.662990F, 0.664138F, 0.665284F, 0.666428F, 0.667571F, 0.668713F, 0.669852F, 0.670990F, 
  0.672127F, 0.673262F, 0.674396F, 0.675527F, 0.676658F, 0.677786F, 0.678913F, 0.680039F, 
  0.681163F, 0.682285F, 0.683406F, 0.684525F, 0.685642F, 0.686758F, 0.687872F, 0.688985F, 
  0.690096F, 0.691205F, 0.692313F, 0.693419F, 0.694524F, 0.695626F, 0.696728F, 0.697827F, 
  0.698925F, 0.700021F, 0.701116F, 0.702209F, 0.703300F, 0.704390F, 0.705478F, 0.706564F, 
  0.707649F, 0.708732F, 0.709813F, 0.710893F, 0.711971F, 0.713047F, 0.714122F, 0.715195F, 
  0.716266F, 0.717336F, 0.718404F, 0.719470F, 0.720535F, 0.721597F, 0.722659F, 0.723718F, 
  0.724776F, 0.725832F, 0.726886F, 0.727939F, 0.728990F, 0.730039F, 0.731086F, 0.732132F, 
  0.733176F, 0.734218F, 0.735259F, 0.736298F, 0.737335F, 0.738370F, 0.739404F, 0.740436F, 
  0.741466F, 0.742494F, 0.743521F, 0.744546F, 0.745569F, 0.746591F, 0.747610F, 0.748628F, 
  0.749644F, 0.750659F, 0.751671F, 0.752682F, 0.753691F, 0.754698F, 0.755704F, 0.756708F, 
  0.757710F, 0.758710F, 0.759708F, 0.760705F, 0.761700F, 0.762693F, 0.763684F, 0.764673F, 
  0.765661F, 0.766647F, 0.767631F, 0.768613F, 0.769593F, 0.770572F, 0.771549F, 0.772524F, 
  0.773497F, 0.774468F, 0.775438F, 0.776405F, 0.777371F, 0.778335F, 0.779297F, 0.780258F, 
  0.781216F, 0.782173F, 0.783128F, 0.784081F, 0.785032F, 0.785981F, 0.786929F, 0.787874F, 
  0.788818F, 0.789760F, 0.790700F, 0.791638F, 0.792575F, 0.793509F, 0.794442F, 0.795372F, 
  0.796301F, 0.797228F, 0.798153F, 0.799076F, 0.799998F, 0.800917F, 0.801835F, 0.802750F, 
  0.803664F, 0.804576F, 0.805486F, 0.806394F, 0.807300F, 0.808205F, 0.809107F, 0.810008F, 
  0.810906F, 0.811803F, 0.812698F, 0.813591F, 0.814482F, 0.815371F, 0.816258F, 0.817143F, 
  0.818026F, 0.818908F, 0.819787F, 0.820665F, 0.821540F, 0.822414F, 0.823285F, 0.824155F, 
  0.825023F, 0.825889F, 0.826753F, 0.827615F, 0.828475F, 0.829333F, 0.830189F, 0.831043F, 
  0.831895F, 0.832746F, 0.833594F, 0.834440F, 0.835285F, 0.836127F, 0.836968F, 0.837806F, 
  0.838643F, 0.839477F, 0.840310F, 0.841140F, 0.841969F, 0.842796F, 0.843620F, 0.844443F, 
  0.845264F, 0.846082F, 0.846899F, 0.847714F, 0.848526F, 0.849337F, 0.850146F, 0.850953F, 
  0.851757F, 0.852560F, 0.853361F, 0.854159F, 0.854956F, 0.855751F, 0.856543F, 0.857334F, 
  0.858123F, 0.858909F, 0.859694F, 0.860476F, 0.861257F, 0.862035F, 0.862812F, 0.863586F, 
  0.864359F, 0.865129F, 0.865898F, 0.866664F, 0.867428F, 0.868190F, 0.868951F, 0.869709F, 
  0.870465F, 0.871219F, 0.871971F, 0.872721F, 0.873469F, 0.874215F, 0.874958F, 0.875700F, 
  0.876440F, 0.877177F, 0.877913F, 0.878646F, 0.879378F, 0.880107F, 0.880834F, 0.881559F, 
  0.882283F, 0.883004F, 0.883723F, 0.884439F, 0.885154F, 0.885867F, 0.886578F, 0.887286F, 
  0.887993F, 0.888697F, 0.889399F, 0.890099F, 0.890797F, 0.891493F, 0.892187F, 0.892879F, 
  0.893569F, 0.894256F, 0.894942F, 0.895625F, 0.896307F, 0.896986F, 0.897663F, 0.898338F, 
  0.899011F, 0.899681F, 0.900350F, 0.901016F, 0.901681F, 0.902343F, 0.903003F, 0.903661F, 
  0.904317F, 0.904971F, 0.905622F, 0.906272F, 0.906919F, 0.907564F, 0.908207F, 0.908848F, 
  0.909487F, 0.910124F, 0.910758F, 0.911391F, 0.912021F, 0.912649F, 0.913275F, 0.913899F, 
  0.914520F, 0.915140F, 0.915757F, 0.916372F, 0.916985F, 0.917596F, 0.918205F, 0.918811F, 
  0.919416F, 0.920018F, 0.920618F, 0.921216F, 0.921812F, 0.922405F, 0.922997F, 0.923586F, 
  0.924173F, 0.924758F, 0.925340F, 0.925921F, 0.926499F, 0.927075F, 0.927649F, 0.928221F, 
  0.928791F, 0.929358F, 0.929923F, 0.930486F, 0.931047F, 0.931606F, 0.932162F, 0.932716F, 
  0.933269F, 0.933818F, 0.934366F, 0.934912F, 0.935455F, 0.935996F, 0.936535F, 0.937072F, 
  0.937606F, 0.938138F, 0.938668F, 0.939196F, 0.939722F, 0.940245F, 0.940766F, 0.941285F, 
  0.941802F, 0.942317F, 0.942829F, 0.943339F, 0.943847F, 0.944353F, 0.944856F, 0.945358F, 
  0.945857F, 0.946353F, 0.946848F, 0.947340F, 0.947830F, 0.948318F, 0.948804F, 0.949287F, 
  0.949768F, 0.950247F, 0.950724F, 0.951199F, 0.951671F, 0.952141F, 0.952609F, 0.953074F, 
  0.953537F, 0.953998F, 0.954457F, 0.954914F, 0.955368F, 0.955820F, 0.956270F, 0.956717F, 
  0.957163F, 0.957606F, 0.958047F, 0.958485F, 0.958921F, 0.959355F, 0.959787F, 0.960217F, 
  0.960644F, 0.961069F, 0.961492F, 0.961912F, 0.962330F, 0.962746F, 0.963160F, 0.963571F, 
  0.963980F, 0.964387F, 0.964792F, 0.965194F, 0.965594F, 0.965992F, 0.966388F, 0.966781F, 
  0.967172F, 0.967560F, 0.967947F, 0.968331F, 0.968713F, 0.969092F, 0.969470F, 0.969845F, 
  0.970217F, 0.970588F, 0.970956F, 0.971322F, 0.971685F, 0.972047F, 0.972406F, 0.972762F, 
  0.973117F, 0.973469F, 0.973819F, 0.974166F, 0.974512F, 0.974855F, 0.975195F, 0.975534F, 
  0.975870F, 0.976204F, 0.976535F, 0.976864F, 0.977191F, 0.977516F, 0.977838F, 0.978158F, 
  0.978476F, 0.978791F, 0.979104F, 0.979415F, 0.979724F, 0.980030F, 0.980334F, 0.980635F, 
  0.980935F, 0.981232F, 0.981526F, 0.981819F, 0.982109F, 0.982396F, 0.982682F, 0.982965F, 
  0.983246F, 0.983524F, 0.983800F, 0.984074F, 0.984346F, 0.984615F, 0.984882F, 0.985146F, 
  0.985408F, 0.985668F, 0.985926F, 0.986181F, 0.986434F, 0.986685F, 0.986933F, 0.987179F, 
  0.987423F, 0.987664F, 0.987903F, 0.988140F, 0.988374F, 0.988607F, 0.988836F, 0.989064F, 
  0.989289F, 0.989511F, 0.989732F, 0.989950F, 0.990166F, 0.990379F, 0.990590F, 0.990799F, 
  0.991006F, 0.991210F, 0.991411F, 0.991611F, 0.991808F, 0.992003F, 0.992195F, 0.992385F, 
  0.992573F, 0.992759F, 0.992942F, 0.993122F, 0.993301F, 0.993477F, 0.993651F, 0.993822F, 
  0.993991F, 0.994158F, 0.994322F, 0.994484F, 0.994644F, 0.994802F, 0.994957F, 0.995109F, 
  0.995260F, 0.995408F, 0.995553F, 0.995697F, 0.995838F, 0.995976F, 0.996113F, 0.996247F, 
  0.996378F, 0.996507F, 0.996634F, 0.996759F, 0.996881F, 0.997001F, 0.997119F, 0.997234F, 
  0.997347F, 0.997457F, 0.997565F, 0.997671F, 0.997774F, 0.997876F, 0.997974F, 0.998071F, 
  0.998165F, 0.998257F, 0.998346F, 0.998433F, 0.998518F, 0.998600F, 0.998680F, 0.998758F, 
  0.998833F, 0.998906F, 0.998976F, 0.999044F, 0.999110F, 0.999174F, 0.999235F, 0.999294F, 
  0.999350F, 0.999404F, 0.999456F, 0.999506F, 0.999553F, 0.999597F, 0.999640F, 0.999680F, 
  0.999717F, 0.999753F, 0.999786F, 0.999816F, 0.999844F, 0.999870F, 0.999894F, 0.999915F, 
  0.999934F, 0.999950F, 0.999964F, 0.999976F, 0.999986F, 0.999993F, 0.999997F, 1.000000F 
};

/*! Long KBD Window for frame length 1024 */
const float OnlyLongWindowKBD[FRAME_SIZE] =
{
  0.000293F, 0.000430F, 0.000547F, 0.000655F, 0.000759F, 0.000861F, 0.000962F, 0.001063F, 
  0.001165F, 0.001268F, 0.001372F, 0.001478F, 0.001586F, 0.001696F, 0.001807F, 0.001921F, 
  0.002037F, 0.002156F, 0.002276F, 0.002400F, 0.002526F, 0.002654F, 0.002785F, 0.002919F, 
  0.003056F, 0.003195F, 0.003337F, 0.003483F, 0.003631F, 0.003782F, 0.003936F, 0.004093F, 
  0.004253F, 0.004417F, 0.004584F, 0.004754F, 0.004927F, 0.005103F, 0.005283F, 0.005466F, 
  0.005653F, 0.005843F, 0.006036F, 0.006234F, 0.006434F, 0.006638F, 0.006846F, 0.007058F, 
  0.007273F, 0.007492F, 0.007715F, 0.007942F, 0.008172F, 0.008407F, 0.008645F, 0.008887F, 
  0.009133F, 0.009384F, 0.009638F, 0.009896F, 0.010159F, 0.010426F, 0.010697F, 0.010972F, 
  0.011252F, 0.011535F, 0.011824F, 0.012116F, 0.012413F, 0.012714F, 0.013020F, 0.013331F, 
  0.013646F, 0.013965F, 0.014290F, 0.014619F, 0.014952F, 0.015291F, 0.015634F, 0.015982F, 
  0.016334F, 0.016692F, 0.017054F, 0.017422F, 0.017794F, 0.018172F, 0.018554F, 0.018942F, 
  0.019334F, 0.019732F, 0.020135F, 0.020544F, 0.020957F, 0.021376F, 0.021800F, 0.022229F, 
  0.022664F, 0.023104F, 0.023550F, 0.024001F, 0.024457F, 0.024920F, 0.025387F, 0.025861F, 
  0.026340F, 0.026824F, 0.027315F, 0.027811F, 0.028313F, 0.028820F, 0.029334F, 0.029853F, 
  0.030378F, 0.030910F, 0.031447F, 0.031990F, 0.032539F, 0.033094F, 0.033655F, 0.034223F, 
  0.034796F, 0.035376F, 0.035962F, 0.036554F, 0.037152F, 0.037757F, 0.038367F, 0.038985F, 
  0.039608F, 0.040238F, 0.040875F, 0.041517F, 0.042167F, 0.042823F, 0.043485F, 0.044154F, 
  0.044829F, 0.045511F, 0.046200F, 0.046896F, 0.047598F, 0.048306F, 0.049022F, 0.049744F, 
  0.050473F, 0.051209F, 0.051952F, 0.052702F, 0.053458F, 0.054221F, 0.054992F, 0.055769F, 
  0.056553F, 0.057344F, 0.058143F, 0.058948F, 0.059760F, 0.060579F, 0.061406F, 0.062240F, 
  0.063080F, 0.063928F, 0.064783F, 0.065646F, 0.066515F, 0.067392F, 0.068276F, 0.069167F, 
  0.070066F, 0.070972F, 0.071885F, 0.072806F, 0.073733F, 0.074669F, 0.075611F, 0.076561F, 
  0.077519F, 0.078484F, 0.079456F, 0.080436F, 0.081424F, 0.082418F, 0.083421F, 0.084431F, 
  0.085448F, 0.086473F, 0.087505F, 0.088545F, 0.089593F, 0.090648F, 0.091711F, 0.092781F, 
  0.093859F, 0.094945F, 0.096038F, 0.097139F, 0.098248F, 0.099364F, 0.100488F, 0.101620F, 
  0.102759F, 0.103906F, 0.105060F, 0.106223F, 0.107393F, 0.108570F, 0.109756F, 0.110949F, 
  0.112150F, 0.113358F, 0.114575F, 0.115799F, 0.117030F, 0.118270F, 0.119517F, 0.120772F, 
  0.122034F, 0.123305F, 0.124583F, 0.125869F, 0.127162F, 0.128463F, 0.129772F, 0.131089F, 
  0.132413F, 0.133745F, 0.135085F, 0.136433F, 0.137788F, 0.139151F, 0.140521F, 0.141900F, 
  0.143285F, 0.144679F, 0.146080F, 0.147489F, 0.148906F, 0.150330F, 0.151761F, 0.153201F, 
  0.154648F, 0.156102F, 0.157564F, 0.159034F, 0.160511F, 0.161996F, 0.163488F, 0.164988F, 
  0.166496F, 0.168010F, 0.169533F, 0.171062F, 0.172600F, 0.174144F, 0.175696F, 0.177256F, 
  0.178822F, 0.180397F, 0.181978F, 0.183567F, 0.185163F, 0.186766F, 0.188377F, 0.189995F, 
  0.191620F, 0.193252F, 0.194892F, 0.196539F, 0.198192F, 0.199853F, 0.201521F, 0.203196F, 
  0.204878F, 0.206568F, 0.208264F, 0.209967F, 0.211677F, 0.213394F, 0.215118F, 0.216848F, 
  0.218586F, 0.220330F, 0.222081F, 0.223839F, 0.225604F, 0.227375F, 0.229153F, 0.230937F, 
  0.232729F, 0.234526F, 0.236331F, 0.238141F, 0.239959F, 0.241782F, 0.243612F, 0.245449F, 
  0.247291F, 0.249140F, 0.250996F, 0.252857F, 0.254725F, 0.256599F, 0.258479F, 0.260365F, 
  0.262257F, 0.264155F, 0.266059F, 0.267969F, 0.269885F, 0.271807F, 0.273735F, 0.275668F, 
  0.277607F, 0.279552F, 0.281502F, 0.283458F, 0.285420F, 0.287387F, 0.289360F, 0.291338F, 
  0.293321F, 0.295310F, 0.297304F, 0.299303F, 0.301308F, 0.303318F, 0.305332F, 0.307352F, 
  0.309377F, 0.311407F, 0.313442F, 0.315482F, 0.317526F, 0.319576F, 0.321630F, 0.323689F, 
  0.325752F, 0.327820F, 0.329893F, 0.331970F, 0.334052F, 0.336138F, 0.338228F, 0.340323F, 
  0.342421F, 0.344524F, 0.346632F, 0.348743F, 0.350858F, 0.352978F, 0.355101F, 0.357228F, 
  0.359359F, 0.361493F, 0.363632F, 0.365774F, 0.367920F, 0.370069F, 0.372221F, 0.374378F, 
  0.376537F, 0.378700F, 0.380866F, 0.383035F, 0.385208F, 0.387384F, 0.389562F, 0.391744F, 
  0.393928F, 0.396116F, 0.398306F, 0.400499F, 0.402695F, 0.404893F, 0.407094F, 0.409297F, 
  0.411503F, 0.413712F, 0.415922F, 0.418135F, 0.420350F, 0.422568F, 0.424787F, 0.427008F, 
  0.429232F, 0.431457F, 0.433684F, 0.435913F, 0.438144F, 0.440376F, 0.442611F, 0.444846F, 
  0.447083F, 0.449322F, 0.451562F, 0.453803F, 0.456045F, 0.458289F, 0.460533F, 0.462779F, 
  0.465026F, 0.467274F, 0.469522F, 0.471772F, 0.474022F, 0.476273F, 0.478524F, 0.480776F, 
  0.483029F, 0.485282F, 0.487535F, 0.489789F, 0.492042F, 0.494296F, 0.496551F, 0.498805F, 
  0.501059F, 0.503313F, 0.505567F, 0.507821F, 0.510074F, 0.512328F, 0.514580F, 0.516833F, 
  0.519085F, 0.521336F, 0.523587F, 0.525837F, 0.528086F, 0.530334F, 0.532582F, 0.534828F, 
  0.537073F, 0.539318F, 0.541561F, 0.543803F, 0.546044F, 0.548283F, 0.550522F, 0.552758F, 
  0.554993F, 0.557227F, 0.559459F, 0.561689F, 0.563917F, 0.566144F, 0.568368F, 0.570591F, 
  0.572812F, 0.575031F, 0.577247F, 0.579461F, 0.581674F, 0.583883F, 0.586091F, 0.588296F, 
  0.590498F, 0.592698F, 0.594895F, 0.597090F, 0.599282F, 0.601471F, 0.603657F, 0.605841F, 
  0.608021F, 0.610198F, 0.612372F, 0.614543F, 0.616711F, 0.618876F, 0.621037F, 0.623195F, 
  0.625350F, 0.627501F, 0.629648F, 0.631792F, 0.633932F, 0.636069F, 0.638202F, 0.640330F, 
  0.642456F, 0.644577F, 0.646694F, 0.648807F, 0.650916F, 0.653021F, 0.655121F, 0.657217F, 
  0.659310F, 0.661397F, 0.663480F, 0.665559F, 0.667633F, 0.669703F, 0.671768F, 0.673829F, 
  0.675884F, 0.677935F, 0.679981F, 0.682023F, 0.684059F, 0.686090F, 0.688116F, 0.690138F, 
  0.692154F, 0.694165F, 0.696170F, 0.698171F, 0.700166F, 0.702156F, 0.704140F, 0.706119F, 
  0.708093F, 0.710061F, 0.712023F, 0.713980F, 0.715931F, 0.717877F, 0.719816F, 0.721750F, 
  0.723678F, 0.725600F, 0.727517F, 0.729427F, 0.731331F, 0.733229F, 0.735122F, 0.737008F, 
  0.738888F, 0.740761F, 0.742629F, 0.744490F, 0.746345F, 0.748194F, 0.750036F, 0.751872F, 
  0.753701F, 0.755524F, 0.757340F, 0.759150F, 0.760953F, 0.762750F, 0.764540F, 0.766323F, 
  0.768099F, 0.769869F, 0.771632F, 0.773389F, 0.775138F, 0.776880F, 0.778616F, 0.780345F, 
  0.782066F, 0.783781F, 0.785489F, 0.787189F, 0.788883F, 0.790569F, 0.792249F, 0.793921F, 
  0.795586F, 0.797244F, 0.798895F, 0.800538F, 0.802174F, 0.803803F, 0.805425F, 0.807039F, 
  0.808646F, 0.810245F, 0.811838F, 0.813422F, 0.815000F, 0.816570F, 0.818132F, 0.819687F, 
  0.821234F, 0.822774F, 0.824306F, 0.825831F, 0.827349F, 0.828858F, 0.830360F, 0.831855F, 
  0.833342F, 0.834821F, 0.836293F, 0.837757F, 0.839213F, 0.840661F, 0.842102F, 0.843535F, 
  0.844961F, 0.846379F, 0.847789F, 0.849191F, 0.850586F, 0.851972F, 0.853351F, 0.854723F, 
  0.856086F, 0.857442F, 0.858790F, 0.860130F, 0.861463F, 0.862787F, 0.864104F, 0.865413F, 
  0.866714F, 0.868008F, 0.869293F, 0.870571F, 0.871841F, 0.873103F, 0.874358F, 0.875605F, 
  0.876843F, 0.878074F, 0.879298F, 0.880513F, 0.881721F, 0.882921F, 0.884113F, 0.885297F, 
  0.886474F, 0.887642F, 0.888803F, 0.889957F, 0.891102F, 0.892240F, 0.893370F, 0.894492F, 
  0.895607F, 0.896714F, 0.897813F, 0.898905F, 0.899989F, 0.901065F, 0.902133F, 0.903194F, 
  0.904248F, 0.905293F, 0.906331F, 0.907362F, 0.908385F, 0.909400F, 0.910408F, 0.911408F, 
  0.912401F, 0.913386F, 0.914364F, 0.915334F, 0.916297F, 0.917253F, 0.918201F, 0.919141F, 
  0.920074F, 0.921000F, 0.921919F, 0.922830F, 0.923734F, 0.924630F, 0.925519F, 0.926401F, 
  0.927276F, 0.928144F, 0.929004F, 0.929858F, 0.930704F, 0.931543F, 0.932375F, 0.933199F, 
  0.934017F, 0.934828F, 0.935632F, 0.936429F, 0.937218F, 0.938001F, 0.938777F, 0.939546F, 
  0.940309F, 0.941064F, 0.941813F, 0.942555F, 0.943290F, 0.944018F, 0.944740F, 0.945455F, 
  0.946164F, 0.946865F, 0.947561F, 0.948249F, 0.948932F, 0.949607F, 0.950277F, 0.950939F, 
  0.951596F, 0.952246F, 0.952890F, 0.953527F, 0.954158F, 0.954783F, 0.955401F, 0.956014F, 
  0.956620F, 0.957220F, 0.957815F, 0.958403F, 0.958985F, 0.959561F, 0.960131F, 0.960695F, 
  0.961253F, 0.961805F, 0.962352F, 0.962892F, 0.963427F, 0.963957F, 0.964480F, 0.964998F, 
  0.965510F, 0.966017F, 0.966518F, 0.967014F, 0.967504F, 0.967988F, 0.968467F, 0.968941F, 
  0.969410F, 0.969873F, 0.970331F, 0.970783F, 0.971231F, 0.971673F, 0.972110F, 0.972542F, 
  0.972969F, 0.973390F, 0.973807F, 0.974219F, 0.974626F, 0.975028F, 0.975425F, 0.975818F, 
  0.976205F, 0.976588F, 0.976966F, 0.977340F, 0.977709F, 0.978073F, 0.978432F, 0.978787F, 
  0.979138F, 0.979484F, 0.979826F, 0.980163F, 0.980496F, 0.980825F, 0.981149F, 0.981469F, 
  0.981785F, 0.982097F, 0.982404F, 0.982708F, 0.983007F, 0.983303F, 0.983594F, 0.983881F, 
  0.984165F, 0.984444F, 0.984720F, 0.984992F, 0.985260F, 0.985525F, 0.985785F, 0.986042F, 
  0.986296F, 0.986545F, 0.986791F, 0.987034F, 0.987273F, 0.987509F, 0.987741F, 0.987970F, 
  0.988195F, 0.988417F, 0.988636F, 0.988851F, 0.989064F, 0.989273F, 0.989479F, 0.989681F, 
  0.989881F, 0.990078F, 0.990271F, 0.990462F, 0.990649F, 0.990834F, 0.991016F, 0.991195F, 
  0.991371F, 0.991544F, 0.991714F, 0.991882F, 0.992047F, 0.992209F, 0.992369F, 0.992526F, 
  0.992680F, 0.992832F, 0.992981F, 0.993128F, 0.993273F, 0.993415F, 0.993554F, 0.993691F, 
  0.993826F, 0.993959F, 0.994089F, 0.994217F, 0.994342F, 0.994466F, 0.994587F, 0.994706F, 
  0.994823F, 0.994938F, 0.995051F, 0.995162F, 0.995271F, 0.995378F, 0.995483F, 0.995585F, 
  0.995687F, 0.995786F, 0.995883F, 0.995978F, 0.996072F, 0.996164F, 0.996254F, 0.996343F, 
  0.996429F, 0.996514F, 0.996598F, 0.996680F, 0.996760F, 0.996838F, 0.996915F, 0.996991F, 
  0.997065F, 0.997137F, 0.997208F, 0.997278F, 0.997346F, 0.997413F, 0.997478F, 0.997542F, 
  0.997605F, 0.997666F, 0.997727F, 0.997785F, 0.997843F, 0.997899F, 0.997954F, 0.998008F, 
  0.998061F, 0.998113F, 0.998163F, 0.998213F, 0.998261F, 0.998308F, 0.998354F, 0.998400F, 
  0.998444F, 0.998487F, 0.998529F, 0.998570F, 0.998610F, 0.998650F, 0.998688F, 0.998725F, 
  0.998762F, 0.998798F, 0.998833F, 0.998867F, 0.998900F, 0.998932F, 0.998964F, 0.998995F, 
  0.999025F, 0.999054F, 0.999083F, 0.999111F, 0.999138F, 0.999164F, 0.999190F, 0.999215F, 
  0.999240F, 0.999264F, 0.999287F, 0.999310F, 0.999332F, 0.999353F, 0.999374F, 0.999394F, 
  0.999414F, 0.999434F, 0.999452F, 0.999470F, 0.999488F, 0.999505F, 0.999522F, 0.999538F, 
  0.999554F, 0.999570F, 0.999585F, 0.999599F, 0.999613F, 0.999627F, 0.999640F, 0.999653F, 
  0.999666F, 0.999678F, 0.999689F, 0.999701F, 0.999712F, 0.999723F, 0.999733F, 0.999743F, 
  0.999753F, 0.999762F, 0.999772F, 0.999780F, 0.999789F, 0.999797F, 0.999805F, 0.999813F, 
  0.999821F, 0.999828F, 0.999835F, 0.999842F, 0.999848F, 0.999855F, 0.999861F, 0.999867F, 
  0.999872F, 0.999878F, 0.999883F, 0.999888F, 0.999893F, 0.999898F, 0.999902F, 0.999907F, 
  0.999911F, 0.999915F, 0.999919F, 0.999923F, 0.999927F, 0.999930F, 0.999933F, 0.999937F, 
  0.999940F, 0.999943F, 0.999946F, 0.999948F, 0.999951F, 0.999954F, 0.999956F, 0.999958F, 
  0.999960F, 0.999963F, 0.999965F, 0.999967F, 0.999968F, 0.999970F, 0.999972F, 0.999974F, 
  0.999975F, 0.999977F, 0.999978F, 0.999979F, 0.999981F, 0.999982F, 0.999983F, 0.999984F, 
  0.999985F, 0.999986F, 0.999987F, 0.999988F, 0.999989F, 0.999990F, 0.999990F, 0.999991F, 
  0.999992F, 0.999992F, 0.999993F, 0.999993F, 0.999994F, 0.999994F, 0.999995F, 0.999995F, 
  0.999996F, 0.999996F, 0.999996F, 0.999997F, 0.999997F, 0.999997F, 0.999998F, 0.999998F, 
  0.999998F, 0.999998F, 0.999999F, 0.999999F, 0.999999F, 0.999999F, 0.999999F, 0.999999F, 
  0.999999F, 1.000000F, 1.000000F, 1.000000F, 1.000000F, 1.000000F, 1.000000F, 1.000000F
};

/*! Short Sine Window for frame length 1024 */
const float OnlyShortWindowSine[FRAME_SIZE/8] =
{
  0.006136F, 0.018407F, 0.030675F, 0.042938F, 0.055195F, 0.067444F, 0.079682F, 0.091909F, 
  0.104122F, 0.116319F, 0.128498F, 0.140658F, 0.152797F, 0.164913F, 0.177004F, 0.189069F, 
  0.201105F, 0.213110F, 0.225084F, 0.237024F, 0.248928F, 0.260794F, 0.272621F, 0.284408F, 
  0.296151F, 0.307850F, 0.319502F, 0.331106F, 0.342661F, 0.354164F, 0.365613F, 0.377007F, 
  0.388345F, 0.399624F, 0.410843F, 0.422000F, 0.433094F, 0.444122F, 0.455084F, 0.465977F, 
  0.476799F, 0.487550F, 0.498228F, 0.508830F, 0.519356F, 0.529804F, 0.540172F, 0.550458F, 
  0.560662F, 0.570781F, 0.580814F, 0.590760F, 0.600617F, 0.610383F, 0.620057F, 0.629638F, 
  0.639124F, 0.648514F, 0.657807F, 0.667000F, 0.676093F, 0.685084F, 0.693971F, 0.702755F, 
  0.711432F, 0.720003F, 0.728464F, 0.736817F, 0.745058F, 0.753187F, 0.761202F, 0.769103F, 
  0.776888F, 0.784557F, 0.792107F, 0.799537F, 0.806848F, 0.814036F, 0.821103F, 0.828045F, 
  0.834863F, 0.841555F, 0.848120F, 0.854558F, 0.860867F, 0.867046F, 0.873095F, 0.879012F, 
  0.884797F, 0.890449F, 0.895966F, 0.901349F, 0.906596F, 0.911706F, 0.916679F, 0.921514F, 
  0.926210F, 0.930767F, 0.935184F, 0.939459F, 0.943593F, 0.947586F, 0.951435F, 0.955141F, 
  0.958703F, 0.962121F, 0.965394F, 0.968522F, 0.971504F, 0.974339F, 0.977028F, 0.979570F, 
  0.981964F, 0.984210F, 0.986308F, 0.988258F, 0.990058F, 0.991710F, 0.993212F, 0.994565F, 
  0.995767F, 0.996820F, 0.997723F, 0.998476F, 0.999078F, 0.999529F, 0.999831F, 0.999981F
} ;

/*! Short KBD Window for frame length 1024 */
const float OnlyShortWindowKBD[FRAME_SIZE/8] =
{
  0.000044F, 0.000119F, 0.000231F, 0.000389F, 0.000606F, 0.000892F, 0.001262F, 0.001730F, 
  0.002314F, 0.003031F, 0.003902F, 0.004947F, 0.006189F, 0.007651F, 0.009360F, 0.011340F, 
  0.013620F, 0.016227F, 0.019190F, 0.022539F, 0.026303F, 0.030512F, 0.035195F, 0.040381F, 
  0.046099F, 0.052375F, 0.059236F, 0.066706F, 0.074808F, 0.083563F, 0.092988F, 0.103099F, 
  0.113909F, 0.125427F, 0.137659F, 0.150608F, 0.164272F, 0.178647F, 0.193722F, 0.209486F, 
  0.225920F, 0.243003F, 0.260711F, 0.279012F, 0.297874F, 0.317259F, 0.337126F, 0.357432F, 
  0.378127F, 0.399163F, 0.420486F, 0.442041F, 0.463771F, 0.485618F, 0.507521F, 0.529420F, 
  0.551256F, 0.572968F, 0.594497F, 0.615784F, 0.636772F, 0.657406F, 0.677634F, 0.697406F, 
  0.716676F, 0.735399F, 0.753536F, 0.771052F, 0.787915F, 0.804098F, 0.819577F, 0.834336F, 
  0.848360F, 0.861640F, 0.874171F, 0.885955F, 0.896995F, 0.907299F, 0.916880F, 0.925754F, 
  0.933939F, 0.941459F, 0.948339F, 0.954605F, 0.960288F, 0.965417F, 0.970025F, 0.974146F, 
  0.977812F, 0.981056F, 0.983913F, 0.986415F, 0.988594F, 0.990480F, 0.992103F, 0.993491F, 
  0.994671F, 0.995667F, 0.996503F, 0.997198F, 0.997773F, 0.998244F, 0.998627F, 0.998937F, 
  0.999184F, 0.999380F, 0.999534F, 0.999654F, 0.999746F, 0.999816F, 0.999868F, 0.999907F, 
  0.999936F, 0.999956F, 0.999971F, 0.999981F, 0.999988F, 0.999992F, 0.999995F, 0.999997F, 
  0.999999F, 0.999999F, 1.000000F, 1.000000F, 1.000000F, 1.000000F, 1.000000F, 1.000000F
};



/* table sin k*pi/1024 */
const float trigData[513] =
{
  0.00000000F, 0.00306796F, 0.00613588F, 0.00920375F,
  0.01227154F, 0.01533921F, 0.01840673F, 0.02147408F,
  0.02454123F, 0.02760815F, 0.03067480F, 0.03374117F,
  0.03680722F, 0.03987293F, 0.04293826F, 0.04600318F,
  0.04906767F, 0.05213170F, 0.05519524F, 0.05825826F,
  0.06132074F, 0.06438263F, 0.06744392F, 0.07050457F,
  0.07356456F, 0.07662386F, 0.07968244F, 0.08274026F,
  0.08579731F, 0.08885355F, 0.09190896F, 0.09496350F,
  0.09801714F, 0.10106986F, 0.10412163F, 0.10717242F,
  0.11022221F, 0.11327095F, 0.11631863F, 0.11936521F,
  0.12241068F, 0.12545498F, 0.12849811F, 0.13154003F,
  0.13458071F, 0.13762012F, 0.14065824F, 0.14369503F,
  0.14673047F, 0.14976453F, 0.15279719F, 0.15582840F,
  0.15885814F, 0.16188639F, 0.16491312F, 0.16793829F,
  0.17096189F, 0.17398387F, 0.17700422F, 0.18002290F,
  0.18303989F, 0.18605515F, 0.18906866F, 0.19208040F,
  0.19509032F, 0.19809841F, 0.20110463F, 0.20410897F,
  0.20711138F, 0.21011184F, 0.21311032F, 0.21610680F,
  0.21910124F, 0.22209362F, 0.22508391F, 0.22807208F,
  0.23105811F, 0.23404196F, 0.23702361F, 0.24000302F,
  0.24298018F, 0.24595505F, 0.24892761F, 0.25189782F,
  0.25486566F, 0.25783110F, 0.26079412F, 0.26375468F,
  0.26671276F, 0.26966833F, 0.27262136F, 0.27557182F,
  0.27851969F, 0.28146494F, 0.28440754F, 0.28734746F,
  0.29028468F, 0.29321916F, 0.29615089F, 0.29907983F,
  0.30200595F, 0.30492923F, 0.30784964F, 0.31076715F,
  0.31368174F, 0.31659338F, 0.31950203F, 0.32240768F,
  0.32531029F, 0.32820984F, 0.33110631F, 0.33399965F,
  0.33688985F, 0.33977688F, 0.34266072F, 0.34554132F,
  0.34841868F, 0.35129276F, 0.35416353F, 0.35703096F,
  0.35989504F, 0.36275572F, 0.36561300F, 0.36846683F,
  0.37131719F, 0.37416406F, 0.37700741F, 0.37984721F,
  0.38268343F, 0.38551605F, 0.38834505F, 0.39117038F,
  0.39399204F, 0.39680999F, 0.39962420F, 0.40243465F,
  0.40524131F, 0.40804416F, 0.41084317F, 0.41363831F,
  0.41642956F, 0.41921689F, 0.42200027F, 0.42477968F,
  0.42755509F, 0.43032648F, 0.43309382F, 0.43585708F,
  0.43861624F, 0.44137127F, 0.44412214F, 0.44686884F,
  0.44961133F, 0.45234959F, 0.45508359F, 0.45781330F,
  0.46053871F, 0.46325978F, 0.46597650F, 0.46868882F,
  0.47139674F, 0.47410021F, 0.47679923F, 0.47949376F,
  0.48218377F, 0.48486925F, 0.48755016F, 0.49022648F,
  0.49289819F, 0.49556526F, 0.49822767F, 0.50088538F,
  0.50353838F, 0.50618665F, 0.50883014F, 0.51146885F,
  0.51410274F, 0.51673180F, 0.51935599F, 0.52197529F,
  0.52458968F, 0.52719913F, 0.52980362F, 0.53240313F,
  0.53499762F, 0.53758708F, 0.54017147F, 0.54275078F,
  0.54532499F, 0.54789406F, 0.55045797F, 0.55301671F,
  0.55557023F, 0.55811853F, 0.56066158F, 0.56319934F,
  0.56573181F, 0.56825895F, 0.57078075F, 0.57329717F,
  0.57580819F, 0.57831380F, 0.58081396F, 0.58330865F,
  0.58579786F, 0.58828155F, 0.59075970F, 0.59323230F,
  0.59569930F, 0.59816071F, 0.60061648F, 0.60306660F,
  0.60551104F, 0.60794978F, 0.61038281F, 0.61281008F,
  0.61523159F, 0.61764731F, 0.62005721F, 0.62246128F,
  0.62485949F, 0.62725182F, 0.62963824F, 0.63201874F,
  0.63439328F, 0.63676186F, 0.63912444F, 0.64148101F,
  0.64383154F, 0.64617601F, 0.64851440F, 0.65084668F,
  0.65317284F, 0.65549285F, 0.65780669F, 0.66011434F,
  0.66241578F, 0.66471098F, 0.66699992F, 0.66928259F,
  0.67155895F, 0.67382900F, 0.67609270F, 0.67835004F,
  0.68060100F, 0.68284555F, 0.68508367F, 0.68731534F,
  0.68954054F, 0.69175926F, 0.69397146F, 0.69617713F,
  0.69837625F, 0.70056879F, 0.70275474F, 0.70493408F,
  0.70710678F, 0.70927283F, 0.71143220F, 0.71358487F,
  0.71573083F, 0.71787005F, 0.72000251F, 0.72212819F,
  0.72424708F, 0.72635916F, 0.72846439F, 0.73056277F,
  0.73265427F, 0.73473888F, 0.73681657F, 0.73888732F,
  0.74095113F, 0.74300795F, 0.74505779F, 0.74710061F,
  0.74913639F, 0.75116513F, 0.75318680F, 0.75520138F,
  0.75720885F, 0.75920919F, 0.76120239F, 0.76318842F,
  0.76516727F, 0.76713891F, 0.76910334F, 0.77106052F,
  0.77301045F, 0.77495311F, 0.77688847F, 0.77881651F,
  0.78073723F, 0.78265060F, 0.78455660F, 0.78645521F,
  0.78834643F, 0.79023022F, 0.79210658F, 0.79397548F,
  0.79583690F, 0.79769084F, 0.79953727F, 0.80137617F,
  0.80320753F, 0.80503133F, 0.80684755F, 0.80865618F,
  0.81045720F, 0.81225059F, 0.81403633F, 0.81581441F,
  0.81758481F, 0.81934752F, 0.82110251F, 0.82284978F,
  0.82458930F, 0.82632106F, 0.82804505F, 0.82976123F,
  0.83146961F, 0.83317016F, 0.83486287F, 0.83654773F,
  0.83822471F, 0.83989379F, 0.84155498F, 0.84320824F,
  0.84485357F, 0.84649094F, 0.84812034F, 0.84974177F,
  0.85135519F, 0.85296060F, 0.85455799F, 0.85614733F,
  0.85772861F, 0.85930182F, 0.86086694F, 0.86242396F,
  0.86397286F, 0.86551362F, 0.86704625F, 0.86857071F,
  0.87008699F, 0.87159509F, 0.87309498F, 0.87458665F,
  0.87607009F, 0.87754529F, 0.87901223F, 0.88047089F,
  0.88192126F, 0.88336334F, 0.88479710F, 0.88622253F,
  0.88763962F, 0.88904836F, 0.89044872F, 0.89184071F,
  0.89322430F, 0.89459949F, 0.89596625F, 0.89732458F,
  0.89867447F, 0.90001589F, 0.90134885F, 0.90267332F,
  0.90398929F, 0.90529676F, 0.90659570F, 0.90788612F,
  0.90916798F, 0.91044129F, 0.91170603F, 0.91296219F,
  0.91420976F, 0.91544872F, 0.91667906F, 0.91790078F,
  0.91911385F, 0.92031828F, 0.92151404F, 0.92270113F,
  0.92387953F, 0.92504924F, 0.92621024F, 0.92736253F,
  0.92850608F, 0.92964090F, 0.93076696F, 0.93188427F,
  0.93299280F, 0.93409255F, 0.93518351F, 0.93626567F,
  0.93733901F, 0.93840353F, 0.93945922F, 0.94050607F,
  0.94154407F, 0.94257320F, 0.94359346F, 0.94460484F,
  0.94560733F, 0.94660091F, 0.94758559F, 0.94856135F,
  0.94952818F, 0.95048607F, 0.95143502F, 0.95237501F,
  0.95330604F, 0.95422810F, 0.95514117F, 0.95604525F,
  0.95694034F, 0.95782641F, 0.95870347F, 0.95957151F,
  0.96043052F, 0.96128049F, 0.96212140F, 0.96295327F,
  0.96377607F, 0.96458979F, 0.96539444F, 0.96619000F,
  0.96697647F, 0.96775384F, 0.96852209F, 0.96928124F,
  0.97003125F, 0.97077214F, 0.97150389F, 0.97222650F,
  0.97293995F, 0.97364425F, 0.97433938F, 0.97502535F,
  0.97570213F, 0.97636973F, 0.97702814F, 0.97767736F,
  0.97831737F, 0.97894818F, 0.97956977F, 0.98018214F,
  0.98078528F, 0.98137919F, 0.98196387F, 0.98253930F,
  0.98310549F, 0.98366242F, 0.98421009F, 0.98474850F,
  0.98527764F ,0.98579751F, 0.98630810F, 0.98680940F,
  0.98730142F ,0.98778414F, 0.98825757F, 0.98872169F,
  0.98917651F ,0.98962202F, 0.99005821F, 0.99048508F,
  0.99090264F ,0.99131086F, 0.99170975F, 0.99209931F,
  0.99247953F ,0.99285041F, 0.99321195F, 0.99356414F,
  0.99390697F ,0.99424045F, 0.99456457F, 0.99487933F,
  0.99518473F ,0.99548076F, 0.99576741F, 0.99604470F,
  0.99631261F ,0.99657115F, 0.99682030F, 0.99706007F,
  0.99729046F ,0.99751146F, 0.99772307F, 0.99792529F,
  0.99811811F ,0.99830154F, 0.99847558F, 0.99864022F,
  0.99879546F ,0.99894129F, 0.99907773F, 0.99920476F,
  0.99932238F ,0.99943060F, 0.99952942F, 0.99961882F,
  0.99969882F ,0.99976941F, 0.99983058F, 0.99988235F,
  0.99992470F ,0.99995764F, 0.99998118F, 0.99999529F,
  1.00000000F
};

/*
  entry for each sampling rate
  * 1  Main/LC long window
  * 2  Main/LC short window
*/
const char tns_max_bands_tbl[9][2] =
{
  { 40, 14 }, /* 48000 */
  { 42, 14 }, /* 44100 */
  { 51, 14 }, /* 32000 */
  { 46, 14 }, /* 24000 */
  { 46, 14 }, /* 22050 */
  { 42, 14 }, /* 16000 */
  { 42, 14 }, /* 12000 */
  { 42, 14 }, /* 11025 */
  { 39, 14 }, /*  8000 */
};

const float tnsCoeff3[8] =
{
  -0.98480775F,
  -0.86602539F,
  -0.64278758F,
  -0.34202015F,
   0.00000000F,
   0.43388373F,
   0.78183150F,
   0.97492790F
};

const float tnsCoeff4[16] =
{
  -0.99573418F,
  -0.96182567F,
  -0.89516330F,
  -0.79801726F,
  -0.67369568F,
  -0.52643222F,
  -0.36124170F,
  -0.18374953F,
   0.00000000F,
   0.20791170F,
   0.40673664F,
   0.58778524F,
   0.74314481F,
   0.86602539F,
   0.95105654F,
   0.99452192F
};

