	page	59,132
	title	UCACHE2 -- DOS Caching Driver.
;
; UCACHE2 is a "caching only" variant of UHDD, offering all UHDD
; features except disk or diskette handling.   UCACHE2 can cache
; 5-Megabytes to 4-GIGABYTES of data!
;
;
; General Program Equations.
;
	.386p			;Allow use of 80386 instructions.
s	equ	<short>		;Make a conditional jump "short".
STACK	equ	432		;Driver local-stack size.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
XMSERR	equ	0FFh		;XMS "memory error" return code.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?		;(Segment must ALWAYS be set!).
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	(I_Stra-@)	;"Strategy" routine pointer.
XMSHndl	dw	(I_Init-@)	;Device-Interrupt routine pointer.
				;(XMS "handle" number, after Init).
	db	'UCACHE2$'	;Driver name, fixed to "UCACHE2$".
;
; General Driver "Data Page" Variables.   "RqSec" through "RqTyp" are
;   located here so caching calls stay compatible with the first UHDD
;   driver of 2-Aug-2012.
;
VLF	db	0		;VDS "lock" flag (001h = buffer lock).
IOF	db	0		;I-O control flags --
				;  Bit 1:  Cache must be "flushed".
				;  Bit 0:  Driver is currently "busy".
CBAdr	dd	0		;User-driver "callback" address.
RqSec	db	0		;Caching request I-O sector count.
RqCmd	db	0		;Caching I-O command bits.
RqCSC	db	0		;Caching current-block sector count.
RqTyp	db	0		;Caching device-type flag.
CStack	dd	0		;Caller's saved stack pointer.
STLmt	dw	0		;Cache binary-search limit index.
LUTop	dw	0FFFFh		;Least-recently-used "top" index.
LUEnd	dw	0FFFFh		;Least-recently-used "end" index.
WrkBf	dw	0		;"LRUPut" working buffer.
RqNdx	dw	0		;Request cache-table index.
TmpBf	dw	0		;Ye olde "temporary" bufffer.
LBABuf	dw	0,0,0,0		;"Calculated LBA" buffer.
RqBuf	dd	0		;Request I-O buffer address.
RqXMS	dd	0		;Request XMS buffer offset.
RqLBA	dw	0,0,0		;Request initial LBA address.
RqUNo	db	0,0		;Request "cache unit" number.
CBLBA	dw	0,0,0		;Current-buffer initial LBA.
CBUNo	db	0		;Current-buffer "cache unit" number.
CBSec	db	0		;Current-buffer sector count.
CBLst	dw	0		;Current-buffer "last" LRU index.
CBNxt	dw	0		;Current-buffer "next" LRU index.
;
; User-Request Entry Point.
;
Entry:	mov	cs:CStack.lw,sp	;Save caller's stack pointer.
	mov	cs:CStack.hw,ss
	push	cs		;Switch to our driver stack, to avoid
	pop	ss		;  CONFIG.SYS "STACKS=0,0" problems!!
	mov	sp,0
@Stack	equ	[$-2].lw	;(Starting stack pointer, Init set).
	sti			;Re-enable CPU interrupts.
	pushad			;Save all CPU registers.
	push	ds
	push	es
	pusha			;Issue "A20 local-enable" request.
	mov	ah,005h
	call	A20Req
	popa
	jnz s	A20Err		;If "A20" error, use routine below.
	db	0EAh		;Go to main Int 13h request handler.
@CMain	dd	(Cache-@)	;(Main entry address, set by Init).
A20Err:	mov	al,XMSERR	;"A20" error!  Post "XMS error" code.
	or	IOF,002h	;Flush cache on next caching request.
	stc			;Set carry flag to report an "error".
	jmp s	Abandn		;Abandon this request and exit QUICK!
CExit:	call	A20Req		;Done -- Issue "A20 local-disable"
	pop	ax		;Reload error code and error flag.
	popf
Abandn:	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+33],al	;Set error code in exiting AH-reg.
	cli			;Disable CPU interrupts.
	dec	IOF		;Reset driver "busy" flag.
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1	;(One of the flags saved by Int 13h).
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-register and CPU flags.
	popf
	iret			;Exit.
;
; User-Request Entry "Vector".   This MUST be at driver address 00A0h
;   for compatibility with the user-driver logic needed to call UHDD.
;
Vector:	jmp s	Entry		;Go to user-request entry routine.
;
; Subroutine to issue "A20" local-enable and local-disable requests.
;
A20Req:	db	09Ah		;Call XMS manager for "A20" request.
@XEntry	dd	0		;(XMS "entry" address, set by Init).
	dec	ax		;Zero AX-reg. if success, -1 if error.
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry flag.
	ret			;Exit.
LMEnd	equ	($+STACK)	;End of our driver's low-memory space.
				;  All that follows can go in the HMA.
;
; Main Caching Routine.
;
Cache:	mov	[bx+RqBuf-@],eax    ;Set user I-O buffer address.
	mov	[bx+RqSec-@],cx	    ;Set sector count & I-O command.
	mov	[bx+RqLBA-@],di	    ;Set initial request LBA.
	mov	[bx+RqLBA+2-@],dx
	mov	[bx+RqLBA+4-@],si
	mov	[bx+RqUNo-@],bp	    ;Set "cache unit" number.
	btr	[bx+IOF-@].lb,1     ;Does cache need to be flushed?
	jnc s	NextIO		    ;No, proceed with this request.
	mov	[bx+STLmt-@],bx	    ;Reset binary-search limit index.
	or	[bx+LUTop-@].dwd,-1 ;Reset LRU start & end indexes.
	xchg	bx,bx		    ;(Unused alignment "filler").
NextIO:	push	ds		;Set ES-reg. same as DS-reg.
	pop	es
	mov	al,0		;Get "granularity" (sectors/block).
@GRAN1	equ	[$-1].lb	;(Block "granularity", set by Init).
	cmp	al,[bx+RqSec-@]	;Will we need multiple cache blocks?
	jbe s	SetSC		;Yes, use "full block" sector count.
	mov	al,[bx+RqSec-@]	;Use remaining request sector count.
SetSC:	mov	[bx+RqCSC-@],al	;Set maximum I-O sector count.
	mov	[bx+RqXMS-@],bx ;Reset lower XMS buffer offset.
	mov	dx,0		;Set initial binary-search offset.
@MP1	equ	[$-2].lw	;(Search-table midpoint, Init set).
	mov	bp,dx		;Set initial binary-search index.
	dec	bp
Search:	shr	dx,1		;Divide binary-search offset by 2.
	cmp	bp,[bx+STLmt-@]	;Is our search index too high?
	jae s	SrchLo		;Yes, cut search index by offset.
	call	STGet		;Get next binary-search table index.
	jc s	SrchE		;If any XMS error, exit immediately!
	mov	ax,[bx+WrkBf-@]	;Get next cache entry into buffer.
	call	CBGet
SrchE:	jc	EndIO		;If any XMS error, exit immediately!
	mov	si,(RqLBA-@)	;Compare initial request & table LBA.
	call	CComp
	jae s	ChkEnd		;Request >= table:  Check for found.
SrchLo:	sub	bp,dx		;Subtract offset from search ptr.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	jmp s	NoFind		;Handle this request as "not found".
ChkEnd:	je	Found		;Request = table:  Treat as "found".
	mov	cl,[bx+CBSec-@]	;Calculate and set ending entry LBA.
	mov	si,(CBLBA-@)
	call	CalcEA
	mov	si,(RqLBA-@)	;Compare request start & entry end.
	call	CComp1
	jb s	Found		  ;Request < Entry:  Handle as found.
	ja s	SrchHi		  ;Request > Entry:  Bump search ptr.
	cmp	[bx+CBSec-@].lb,0 ;Is this cache entry "full"?
@GRAN2	equ	[$-1].lb	  ;(Block "granularity", Init set).
	jb s	Found		;No, handle this request as "found".
SrchHi:	add	bp,dx		;Add offset to search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	inc	bp		;Bump index to next table entry.
NoFind:	mov	[bx+RqNdx-@],bp	;Unfound:  Save search-table offset.
	mov	bp,[bx+STLmt-@]	;Get next "free" cache-table index
	call	STGet		;  and leave index in work buffer.
	jc s	NFind1		;If any XMS error, exit immediately!
	movzx	ecx,bp		  ;Get move-up word count.
	movzx	esi,[bx+RqNdx-@].lw
	sub	cx,si		  ;Any search indexes to move up?
	jz s	NFind2		  ;No, go set up new cache entry.
	shl	ecx,1		  ;Set move-up byte count.
	shl	esi,1		  ;Set 32-bit move-up addresses.
	add	esi,080000000h
@STAdr1	equ	[$-4].dwd
	lea	edi,[esi+2]
	push	ecx		;Save move length & destination addr.
	push	edi
	mov	edi,0		;Set search-buffer destination addr.
@SBuff	equ	[$-4].dwd
	push	edi		;Save search-table buffer address.
	call	MvData		;Send needed data to search buffer.
	pop	esi		;Reload search-table buffer address.
	pop	edi		;Reload move destination & length.
	pop	ecx
	jc s	NFind1		;If any XMS error, exit immediately!
	call	MvData		;Bring data BACK from search buffer.
	jc s	NFind1		;If any XMS error, exit immediately!
	mov	bp,[bx+RqNdx-@]	;Set up table-index "put" to XMS.
	inc	bx
	call	STPut		;Insert "free" index in search table.
	nop			;(Unused alignment "filler").
NFind1:	jc s	UpdErr		;If XMS error, exit immediately!
NFind2:	inc	[bx+STLmt-@].lw	;Advance binary-search limit index.
	mov	si,(RqLBA-@)	;Set 48-bit LBA in new entry.
	mov	di,(CBLBA-@)
	movsd
	movsw
	movsb			;Set "cache unit" in new entry.
	mov	[di],bl		;Reset new entry's sector count.
	mov	ax,[bx+WrkBf-@]	;Reload "free" cache-table index.
Found:	mov	[bx+RqNdx-@],ax	   ;Post current cache-entry index.
	mov	cx,[bx+RqLBA+4-@]  ;Get starting I-O offset in block.
	sub	cx,[bx+CBLBA+4-@]
	mov	[bx+RqXMS-@],cl	   ;Set starting XMS sector offset.
	mov	[bx+TmpBf-@],cx    ;Save starting I-O sector offset.
	cmp	[bx+CBSec-@],bl	   ;Is this a new cache-table entry?
	je s	ReLink		   ;Yes, relink entry as top-of-list.
	push	ax		   ;Unlink this entry from LRU list.
	call	UnLink
	pop	ax
ReLink:	mov	[bx+RqXMS+2-@],bx  ;Reset upper XMS buffer offset.
	movzx	edx,ax		   ;Get 32-bit cache block number.
	shl	edx,0		   ;Shift number to starting sector.
@GRSSC	equ	[$-1].lb	   ;("Granularity" shift, Init set).
	add	[bx+RqXMS-@],edx   ;Add to "preset" sector offset.
	shl	[bx+RqXMS-@].dwd,9	  ;Convert sectors to bytes.
	add	[bx+RqXMS-@].dwd,020000h  ;Add in XMS "base" address.
@XBase	equ	[$-4].dwd		  ;(XMS "base", set by Init).
	mov	cx,0FFFFh	;Make this entry "top of list".
	or	CBLst,cx	;Set this entry's "last" index.
	mov	dx,ax		;Swap top-of-list & entry index.
	xchg	dx,[bx+LUTop-@]
	mov	CBNxt,dx	;Set this entry's "next" index.
	cmp	dx,cx		;Is this the only LRU index?
	mov	cx,ax		;(Get this entry's index in CX-reg.).
	je s	ReLnk1		;Yes, make entry last on LRU list.
	push	ax		;Link entry to prior "top of list".
	call	UnLnk3
	pop	ax
	jmp s	ReLnk2		;Go deal with I-O sector count.
Update:	mov	[bx+CBSec-@],cl	;Update this entry's total sectors.
	call	CBPut		;Update this cache-table entry.
UpdErr:	jc s	EndIOJ		;If any XMS error, exit immediately!
	movzx	cx,[bx+RqCSC-@]	;Calculate ending LBA for this I-O.
	mov	si,(RqLBA-@)
	call	CalcEA
	inc	bp		;Skip to next search index.
Ovrlap:	cmp	bp,[bx+STLmt-@]	;More cache-table entries to check?
	jae s	CachIO		;No, O.K. to handle "found" entry.
	call	STGet		;Get next search table index.
	jc s	EndIOJ		;If any XMS error, bail out now!
	mov	ax,[bx+WrkBf-@]	;Get next cache entry into buffer.
	call	CBGet
	jc s	EndIOJ		;If any XMS error, exit immediately!
	mov	si,(LBABuf-@)	;Compare request end & entry start.
	call	CComp
	jbe s	CachIO		;Request <= entry:  O.K. to proceed.
	push	bp		;Delete this overlapping table entry.
	call	Delete
	pop	bp
	jmp s	Ovrlap		;Go check for more entry overlap.
ReLnk1:	mov	[bx+LUEnd-@],ax	;Make entry last on LRU list, too!
ReLnk2:	mov	cx,[bx+TmpBf-@] ;Reload initial I-O sector offset.
	mov	ch,0		;Get entry's available sectors.
@GRAN3	equ	[$-1].lb	;(Block "granularity", Init set).
	sub	ch,cl
	cmp	ch,[bx+RqCSC-@]	;More I-O sectors than available?
	jae s	Larger		;No, retain maximum sector count.
	mov	[bx+RqCSC-@],ch	;Reduce current I-O sector count.
Larger:	add	cl,[bx+RqCSC-@]	;Get ending I-O sector number.
	cmp	cl,[bx+CBSec-@]	;More sectors than entry has now?
	ja s	Update		;Yes, update entry sectors.
	inc	bx		;Reset Z-flag for "put" into XMS.
	call	CBPut		;Update this cache-table entry.
EndIOJ:	jc s	EndIO		      ;If any XMS error, exit fast!
	test	[bx+RqCmd-@].lb,002h  ;Is this a read request?
	jz s	BufMov		      ;Yes, move cache data to user.
CachIO:	bts	[bx+RqCmd-@].lb,6     ;I-O done during a prior block?
	jc s	BfMore		      ;Yes, buffer more cache data.
	call	[bx+CBAdr-@].dwd ;Call user driver's "callback" rtn.
	sti			 ;Restore critical driver settings.
	cld
	push	08000h		;Reload this driver's DS- & ES-regs.
@DSReg	equ	[$-2].lw	;(Driver segment address, Init set).
	pop	ds
	push	ds
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry.
	jnc s	BfMore		;If no error, go cache this data.
IOErr:	push	ax		;Error!  Save returned error code.
	mov	ax,RqNdx	;Delete cache entry for this I-O.
	call	SrchD
	pop	ax		;Reload returned error code.
	stc			;Set carry again to denote error.
EndIO:	pushf			;Done -- save error flag/code & set
	push	ax		;  "A20 local-disable" command (no
	mov	ah,006h		;  space for these in "base page").
	db	0EAh		;Return to hard-disk exit logic with
	dw	(CExit-@)	;  a 5-byte "jmp" (permits HMA use!).
@CXSeg	dw	0		;(Driver segment address, Init set).
BfMore:	or	ax,1		     ;Ensure we LOAD data into cache!
BufMov:	movzx	ecx,[bx+RqCSC-@].lb  ;Set XMS move sector count.
	mov	esi,[bx+RqXMS-@]     ;Set desired XMS buffer address.
	mov	edi,[bx+RqBuf-@]     ;Set user buffer as destination.
	call	MovBuf		     ;Move data between user and XMS.
	jc s	IOErr			;If error, use routine above.
ChkFul:	cmp	[bx+STLmt-@].lw,08000h  ;Is binary-search table full?
@TE1	equ	[$-2].lw		;(Table "end", Init set).
	jb s	MoreIO		   ;No, check for more sectors to go.
	push	ax		   ;Delete least-recently-used entry.
	mov	ax,[bx+LUEnd-@]
	call	SrchD
	pop	ax
MoreIO:	xor	ax,ax		   ;Reset error code for exit above.
	movzx	cx,[bx+RqCSC-@]	   ;Get current I-O sector count.
	sub	[bx+RqSec-@],cl	   ;More data sectors left to handle?
	jz s	EndIO		   ;No, return to our "exit" routine.
	add	[bx+RqLBA+4-@],cx  ;Update current LBA for next I-O.
	adc	[bx+RqLBA+2-@],bx
	adc	[bx+RqLBA-@],bx
	shl	cx,1		   ;Convert sector count to bytes.
	add	[bx+RqBuf+1-@],cl  ;Update user I-O buffer address.
	adc	[bx+RqBuf+2-@],bx
	jmp	NextIO		   ;Go handle next cache data block.
;
; Subroutine to calculate ending request or cache-table LBA values.
;
CalcEA:	mov	di,(LBABuf-@)	;Point to address-comparison buffer.
	push	[si].dwd	;Move high-order LBA into buffer.
	pop	[di].dwd
	add	cx,[si+4]	;Calculate ending LBA.
	mov	[di+4],cx
	adc	[di+2],bx
	adc	[di],bx
CalcX:	ret			;Exit.
;
; Subroutine to do 7-byte "cache unit" and LBA comparisons.
;
CComp:	mov	di,(CBLBA-@)	;Point to current-buffer LBA value.
CComp1:	mov	cl,[bx+RqUNo-@]	;Compare our "unit" & table "unit".
	cmp	cl,[bx+CBUNo-@]
	jne s	CalcX		;If units are different, exit now.
	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
	ret			;Exit -- Main routine checks results.
;
; Subroutine to "put" an LRU index into its cache-data table entry.
;
LRUPut:	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag (always 0 = "put").
	mov	edi,(WrkBf-@)	;Set our "working" buffer address.
@WrkBf1	equ	[$-4].dwd	;(Working-buffer address, Init set).
	mov	ecx,12		;Set cache-table entry size.
	mul	cx		;Multiply cache index by entry size.
	mov	cl,2		;Set LRU-index size ("put" 2 bytes).
	jmp s	CEMov1		;Go get desired cache-entry offset.
;
; Subroutine to "get" or "put" a cache-data table entry (12 bytes).
;
CBGet:	xor	di,di		;Entry buffer:  Set Z-flag for "get".
CBPut:	mov	edi,(CBLBA-@)	;Set 32-bit "current" buffer address.
@CBAddr	equ	[$-4].dwd	;(Current-buffer address, Init set).
CEMov:	mov	esi,dword ptr 0	;Set cache-table "base" address.
@CTBas1	equ	[$-4].dwd	;(Cache-table "base", Init set).
	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	mov	ecx,12		;Set cache-table entry size.
	mul	cx		;Multiply cache index by entry size.
CEMov1:	push	dx		;Get cache-entry offset in EAX-reg.
	push	ax
	pop	eax
	jmp s	CeMov2		;Go set final XMS data address.
;
; Subroutine to "get" or "put" a binary-search table index (2 bytes).
;
STGet:	xor	di,di		;Search index:  Set Z-flag for get.
STPut:	mov	edi,(WrkBf-@)	;Set 32-bit "working" buffer addr.
@WrkBf2	equ	[$-4].dwd	;(Working-buffer address, Init set).
	mov	esi,0		;Add starting search-table address.
@STAdr2	equ	[$-4].dwd
	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	mov	ecx,2		;Set binary-search index size.
	movzx	eax,bp		;Get 32-bit offset of requested
	shl	eax,1		;  index in binary-search table.
	nop			;(Unused alignment "filler").
CeMov2:	add	esi,eax		;Set XMS data address (base + offset).
	popf			;Reload CPU flags.
	jz s	CeMov3		;Are we "getting" from XMS memory?
	xchg	esi,edi		;No, "swap" source and destination.
CeMov3:	push	bp		;Save BP-reg. now (helps alignment).
	call	MvData		;"Get" or "put" desired cache entry.
	pop	bp		;Reload all 16-bit regs. we used.
	pop	dx
	pop	cx
	pop	ax
	jc s	CEMErr		;If XMS error, post "flush" & exit.
	ret			;All is well -- exit.
CEMErr:	or	[bx+IOF-@].lb,2	;BAD News!  Post "flush cache" flag.
	stc			;Set carry again after "or" command.
	mov	al,XMSERR	;Post "XMS error" return code.
	ret			;Exit.
;
; Global Descriptor Table Pointer, for our real-mode "lgdt" command.
;   This and its following "GDT" are located here for alignment.
;
GDTP	dw	GDTLen		;GDT length (always 56 bytes).
GDTPAdr	dd	(GDT-@)		;GDT 32-bit address (Set By Init).
	db	0,0		;(Unused alignment "filler").
;
; Global Descriptor Table, for cache XMS-memory moves.    Its "code"
;   segment descriptor is unused by the CPU since our real-mode move
;   logic is all "in line", with no "JMP" or "CALL" commands.   Many
;   Thanks to Japheth, for confirming this "poorly documented" info!
;
GDT	dd	0,0		      ;"Null" descriptor.
	dd	0,0		      ;"Code" segment descriptor.
SrcDsc	dd	00000FFFFh,000009300h ;Protected-mode source and
DstDsc	dd	00000FFFFh,000009300h ;  destination descriptors.
	dd	0,0,0,0		      ;(Reserved for use by BIOS).
GDT_DS	dd	00000FFFFh,000CF9300h ;4-GB real-mode descriptor.
GDTLen	equ	($-GDT)		      ;GDT length for real-mode.
;
; Subroutine to search for a cache-table index, and then delete it.
;   Our binary-search table has unit/LBA order, not LRU order!   To
;   delete an LRU index, we do a binary-search for it and then pass
;   the ending BP-reg. value to our "Delete" routine below.
;
SrchD:	call	CBGet		;Get target cache-table entry.
	jc s	SrchDE		;If XMS error, go bail out below!
	mov	si,(CBLBA-@)	;Save target LBA address & unit no.
	mov	di,(LBABuf-@)
	movsd
	movsd
	mov	dx,0		;Set initial binary-search offset.
@MP2	equ	[$-2].lw	;(Search-table midpoint, Init set).
	mov	bp,dx		;Set initial binary-search index.
	dec	bp
SrchD1:	shr	dx,1		;Divide binary-search offset by 2.
	cmp	bp,[bx+STLmt-@]	;Is our search pointer too high?
	jae s	SrchD4		;Yes, cut search pointer by offset.
	call	STGet		;Get next binary-search table index.
	jc s	SrchDE		;If XMS error, go bail out below!
	mov	ax,[bx+WrkBf-@]	;Get next cache entry into buffer.
	call	CBGet
	jc s	SrchDE		;If XMS error, go bail out below!
	mov	si,(LBABuf-@)	;Set up target v.s. work comparison.
	mov	di,(CBLBA-@)
	mov	cl,[si+6]	;Compare target unit v.s. work unit.
	cmp	cl,[di+6]
	jne s	SrchD3		;If units differ, check results now.
	mov	cx,3		;Compare target LBA v.s. work LBA.
	rep	cmpsw
	je s	Delete		;Target = entry:  BP has our offset.
SrchD3:	ja s	SrchD5		;Target > entry:  Adjust offset up.
SrchD4:	sub	bp,dx		;Subtract offset from search ptr.
	jmp s	SrchD6		;Go see if we did our last compare.
SrchD5:	add	bp,dx		;Add offset to search pointer.
SrchD6:	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	SrchD1		;No, go compare next table entry.
	call	CEMErr		;Not found!  Handle as an XMS error!
SrchDE:	jmp s	UnLnkE		;Go discard stack data and exit.
	db	0		;(Unused alignment "filler").
;
; Subroutine to delete a cache index from our search table.
;
Delete:	mov	[bx+WrkBf-@],ax	     ;Save table index being deleted.
	movzx	ecx,[bx+STLmt-@].lw  ;Get move-down word count.
	sub	cx,bp
	dec	cx		;Any other indexes to move down?
	jz s	Delet1		;No, put our index in "free" list.
	shl	ecx,1		;Set move-down byte count.
	movzx	edi,bp		;Set 32-bit move-down addresses.
	shl	edi,1
	add	edi,080000000h
@STAdr3	equ	[$-4].dwd
	lea	esi,[edi+2]
	call	MvData		;Move down needed search-table data.
	jc s	UnLnkE		;If any XMS error, go exit below.
Delet1:	dec	[bx+STLmt-@].lw	;Decrement search-table limit index.
	mov	bp,[bx+STLmt-@]	;Set up table-index "put" to XMS.
	inc	bx
	call	STPut		;Put deleted index in "free" list.
	jc s	UnLnkE		;If any XMS error, go exit below.
;
; Subroutine to unlink a cache-table entry from the LRU list.
;
UnLink:	push	CBLst.dwd	;Get entry's "last"/"next" indexes.
	pop	cx
	pop	dx
	cmp	cx,-1		 ;Is this entry top-of-list?
	je s	UnLnk1		 ;Yes, "promote" next entry.
	mov	[bx+WrkBf-@],dx	 ;Save this entry's "next" index.
	mov	ax,cx		 ;Get "last" entry cache-table index.
	mov	esi,dword ptr 10 ;Get cache-table addr. + "next" ofs.
@CTBas2	equ	[$-4].dwd	 ;(Cache-table address, set by Init).
	call	LRUPut		;Update last entry's "next" index.
	jnc s	UnLnk2		;If no XMS error, check end-of-list.
UnLnkE:	pop	cx		;XMS error!  Discard exit address &
	pop	cx		;  AX/BP parameter saved upon entry.
	jmp	EndIO		;Go post XMS error code and get out!
UnLnk1:	mov	[bx+LUTop-@],dx	;Make next entry top-of-list.
UnLnk2:	cmp	dx,-1		;Is this entry end-of-list?
	jne s	UnLnk3		;No, link next to prior entry.
	mov	[bx+LUEnd-@],cx	;Make prior entry end-of-list.
	ret			;Exit.
UnLnk3:	mov	[bx+WrkBf-@],cx	;Save this entry's "last" index.
	mov	ax,dx		;Get "next" entry cache-table index.
	mov	esi,dword ptr 8 ;Get cache-table addr. + "last" ofs.
@CTBas3	equ	[$-4].dwd	;(Cache-table address, set by Init).
	call	LRUPut		;Update next entry's "last" index.
	jc s	UnLnkE		;If any XMS error, go exit above!
	ret			;All is well -- exit.
;
; Subroutine to do "internal" XMS buffer moves.   This logic was
;   within "UdmaIO" but is now separate, so "UdmaIO"/"DoDMA" may
;   be dismissed (not loaded), if the /E or /N1 switch is given.
;
MovBuf:	jz s	MovBf2		;Is this a read from XMS memory?
	xchg	esi,edi		;No, "swap" source & destination.
	nop			;(Unused alignment "filler").
MovBf2:	shl	ecx,9		;Convert sectors to byte count.
;
; Subroutine to do XMS moves.   At entry, the move parameters are:
;
;   ECX:   Number of bytes to move.   MUST be an even value as only
;	     whole 16-bit words are moved!   Up to a 4-GB value can
;	     be used as data is moved in 2K sections, which permits
;	     interrupts between sections.
;   ESI:   32-bit source data address (NOT segment/offset!).
;   EDI:   32-bit destination address (NOT segment/offset!).
;
; At exit, the carry bit is zero for no errors or is SET for a move
; error.   If so, the AL-reg. has a 0FFh "XMS error" code.   The DS
; and ES-regs. are saved/restored, and the BX-reg. is zero at exit.
; All other registers are NOT saved, for speed.   Only a "protected
; mode" move (JEMM386, etc.) can post an error.   "Real mode" moves
; (UMBPCI, etc.) have NO errors to post!   Our "stand alone" driver
; omits this subroutine and calls the XMS manager to do XMS moves!
;
MvData:	push	ds		;Save driver's segment registers.
	push	es
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands.
	mov	edx,ecx		;Save byte count in EDX-reg.
	mov	ecx,2048	;Get 2K real-mode section count.
	smsw	ax		;Get CPU "machine status word".
	shr	ax,1		;Are we running in protected-mode?
	jc s	MvProt		;Yes, go use protected-mode logic.
MvRNxt:	cmp	ecx,edx		;At least 2048 bytes left?
	cli			;(Disable CPU interrupts for move).
	jbe s	MvRGo		;Yes, use full section count.
	mov	cx,dx		;Use remaining byte count.
MvRGo:	db	02Eh,00Fh	;"lgdt cs:GDTP", coded the hard-way
	db	001h,016h	;   to avoid any annoying V5.1 MASM
	dw	(GDTP-@)	;   warning messages about it!
@GDTP	equ	[$-2].lw
	mov	eax,cr0		;Set CPU protected-mode control bit.
	or	al,001h
	mov	cr0,eax
	mov	bx,(GDT_DS-GDT)	;Set DS and ES "selectors".
	mov	ds,bx
	mov	es,bx
	shr	cx,2		;Convert byte count to dword count.
	db	0F3h,067h	;Move all 32-bit words using "rep"
	movsd			;  and "address-override" prefixes.
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	adc	cx,cx		;If "odd" 16-bit word, move it also,
	db	0F3h,067h	;  using "rep" & "address-override".
 	movsw
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	dec	ax		;Clear protected-mode control bit.
	mov	cr0,eax
	sti			;Allow interrupts after next command.
	mov	cx,2048		;Reload 2K move section count.
	sub	edx,ecx		;Any more data sections to move?
	ja s	MvRNxt		;Yes, go move next data section.
MvDone:	xor	ax,ax		;Success!  Reset carry and error code.
MvExit:	pop	es		;Reload driver's segment registers.
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	ret			;Exit.
MvProt:	shl	ecx,5		;Protected-mode:  Get 64K section ct.
@MvP2K	equ	[$-1].lb	;(If /N4 /Z, "shl ecx,0" = 2K count).
MvPNxt:	push	ecx		;Save move section count.
	push	edx		;Save remaining move byte count.
	push	esi		;Save move source & destination addrs.
	push	edi
	push	cs		;Point ES-reg. to move descriptors.
	pop	es
	cmp	ecx,edx		;At least one section left?
	jbe s	MvPGo		;Yes, use full section count.
	mov	ecx,edx		;Use remaining byte count instead.
MvPGo:	shr	ecx,1		;Convert byte count to word count.
	push	edi		;Set up destination descriptor.
	mov	di,(DstDsc+2-@)
@MvDesc	equ	[$-2].lw
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	push	esi		;Set up source descriptor ("sub"
	sub	di,11		;  zeros carry for our Int 15h).
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	lea	si,[di-21]	;Point to start of descriptor table.
	mov	ah,087h		;Have JEMM386/BIOS move next section.
	int	015h
	pop	edi		;Reload all 32-bit move parameters.
	pop	esi
	pop	edx
	pop	ecx
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands.
	mov	al,XMSERR	;Get our XMS error code.
	jc s	MvExit		;If any BIOS error, exit immediately.
	sub	edx,ecx		;Any more data sections to move?
	jbe s	MvDone		;No, go set "success" code and exit.
	add	esi,ecx		;Update source and dest. addresses.
	add	edi,ecx
	jmp s	MvPNxt		;Go move next section of data.
	db	0		;(Unused alignment "filler").
HMALEN	equ	($-Cache)	;(Length of all HMA caching logic).
;
; Initialization Variables.
;
InitPkt	dd	0		;DOS "Init" packet address.
XMSSize	dd	128		;Total required XMS memory.
HMASize	dw	HMALEN		;Total HMA space required.
SFlag	db	CSDflt		;User cache-size flag.
HFlag	db	0		;"Use HMA space" flag.
;
; Initialization VDS Parameter Block.
;
VDSLn	dd	(LMEnd-@)	;VDS block length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
VDSAd	dd	0		;VDS 32-bit buffer address.
;
; Initialization Cache-Sizes Table.
;
CachSiz	dw	(5*128)		; 5-MB "tiny" cache.
	db	16		;   8K sectors per cache block
	db	4		;      and sector-shift count.
	dw	512		;   512 binary-search midpoint.
	dw	10		;   10K cache-tables XMS memory.
	db	"   5"		;   Title message cache-size text.
	dw	(15*64)		;15-MB "small" cache.
	db	32,5		;   16K granularity values.
	dw	512		;   512 binary-search midpoint.
	dw	15		;   15K cache-tables XMS memory.
	db	"  15"		;   Title message cache-size text.
	dw	(25*64)		;25-MB "medium 1" cache.
	db	32,5		;   16K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	25		;   25K cache-tables XMS memory.
	db	"  25"		;   Title message cache-size text.
	dw	(40*32)		;40-MB "medium 2" cache.
	db	64,6		;   32K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	20		;   20K cache-tables XMS memory.
	db	"  40"		;   Title message cache-size text.
	dw	(50*32)		;50-MB "medium 3" cache.
	db	64,6		;   32K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	25		;   25K cache-tables XMS memory.
	db	"  50"		;   Title message cache-size text.
LCBlks	dw	0FFFFh		;80-MB to 4093-MB "large" cache.
	db	128,7		;   64K granularity values.
	dw	1024		;   1024 to 32768 search midpoint.
	dw	32		;   32K to 1152K cache-tables XMS.
	db	"  80"		;   Title message cache-size text.
CSDflt	equ	5		;Default cache-size flag if no /S.
CSMax	equ	4093		;Maximum cache size in megabytes.
;
; Initialization Messages.
;
TTLMsg	db	CR,LF,'UCACHE2, 8-20-2013.   '
TTL2	db	CR,LF,'$ -MB Cache.',CR,LF,'$'
NXMsg	db	'No XMS$'
VEMsg	db	'VDS init error$'
CPUMsg	db	'No 386+ CPU'
Suffix	db	'; UCACHE2 not loaded!',CR,LF,'$'
;
; Initialization "Strategy" Routine.   This MUST be placed above all
;   run-time logic, to prevent CPU cache "code modification" ERRORS!
;
I_Stra:	mov	cs:InitPkt.lw,bx ;Save DOS request-packet address.
	mov	cs:InitPkt.hw,es
	retf			 ;Exit and await "Device Interrupt".
;
; Initialization "Device Interrupt" Routine.   This is the main init
;   routine for the driver.
;
I_Init:	pushf			;Entry -- save CPU flags.
	push	ds		;Save CPU segment registers.
	push	es
	push	ax		;Save needed 16-bit CPU registers.
	push	bx
	push	dx
	xor	ax,ax		;Get a zero for following logic.
	lds	bx,cs:InitPkt	;Point to DOS "Init" packet.
	cmp	[bx].RPOp,al	;Is this really an "Init" packet?
	jne s	I_Exit			;No?  Reload regs. and exit!
	mov	[bx].RPStat,RPDON+RPERR	;Set "Init" packet defaults
	mov	[bx].RPSeg,cs		;  and "null" driver length.
	and	[bx].RPLen,ax
	push	cs		;NOW point DS-reg. to this driver!
	pop	ds
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax		;(80286+ push SP, then decrement it).
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes?  CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_Sv32		;Yes, go save 32-bit CPU registers.
I_Junk:	mov	dx,(CPUMsg-@)	;Point to "No 386+ CPU" message.
I_Quit:	call	I_Msg		;Display "No 386+" or msg. suffix.
I_Exit:	jmp	I_Bye		;Go reload 16-bit regs. and exit.
I_VErr:	mov	VEMsg.dwd,eax	;Set prefix in "VDS init error" msg.
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_XDis:	mov	dx,CStack.lw	;Load our XMS "handle" number.
	or	dx,dx		;Have we reserved any XMS memory?
	jz s	I_LDMP		;No, reload pointer & display msg.
	mov	ah,00Dh		;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_LDMP:	pop	dx		;Reload error message pointer.
I_EMsg:	call	I_Msg		;Display desired error message.
	popad			;Reload all 32-bit CPU registers.
	mov	dx,(Suffix-@)	;Display message suffix and exit!
	jmp s	I_Quit
I_Sv32:	pushad			;Save all 32-bit CPU registers.
	les	si,es:[bx].RPCL	;Get command-line data pointer.
	xor	bx,bx		;Zero BX-reg. for relative logic.
I_NxtC:	cld			;Ensure FORWARD "string" commands!
	lods	es:[si].lb	;Get next command byte and bump ptr.
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_TrmJ		;Yes, go validate desired cache size.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_TrmJ		;Yes, go validate desired cache size.
	cmp	al,CR		;Is byte an ASCII carriage-return?
I_TrmJ:	je	I_Term		;Yes, go validate desired cache size.
	cmp	al,'/'		;Is byte a slash?
	je s	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,'-'		;Is byte a dash?
	jne s	I_NxtC		;No, check next command-line byte.
I_NxtS:	mov	ax,es:[si]	;Get next 2 command-line bytes.
	and	al,0DFh		;Mask out 1st byte's lower-case bit.
	cmp	al,'H'		;Is switch byte an "H" or "h"?
	jne s	I_CkN4		;No, see if 2 bytes are "N4"/"n4".
	mov	HFlag,al	;Set "use HMA space" flag.
I_CkN4:	cmp	ax,"4N"		;Are these 2 switch bytes "N4"?
	je s	I_ChkZ		;Yes, enable 2K protected XMS moves.
	cmp	al,'Z'		;Is 1st switch byte a "Z" or "z"?
	jne s	I_ChkR		;No, see if byte is "R" or "r".
I_ChkZ:	mov	@MvP2K,bl	;Do slower but "safe" 2K XMS moves.
I_ChkR:	cmp	al,'R'		;Is switch byte an "R" or "r"?
	jne s	I_ChkS		;No, see if byte is "S" or "s".
	mov	ax,es:[si+1]	;Get next 2 command-line bytes.
	mov	cx,15296	;Get 15-MB XMS memory size.
	cmp	ax,"51"		;Does user want 15-MB XMS reserved?
	je s	I_CkRA		;Yes, set memory size to reserve.
	mov	ch,(64448/256)	;Get 63-MB XMS memory size.
	cmp	ax,"36"		;Does user want 63-MB XMS reserved?
	jne s	I_NxtJ		;No, continue scan for a terminator.
I_CkRA:	mov	CStack.hw,cx	;Set desired XMS memory to reserve.
I_NxtJ:	jmp	I_NxtC		;Continue scanning for a terminator.
I_ChkS:	cmp	al,'S'		;Is switch byte an "S" or "s"?
	jne s	I_NxtJ		;No, continue scan for a terminator.
	mov	di,(LCBlks-@)	  ;Point to "large cache" block ct.
	mov	[di+8].dwd,"    " ;Reset "large cache" title bytes.
I_CkS0:	mov	[di].lw,08000h	  ;Invalidate cache-block count.
I_CkS1:	inc	si		;Bump ptr. past "S" or last digit.
	movzx	ax,es:[si].lb	;Get next command-line byte.
	cmp	al,'9'		;Is byte greater than a '9'?
	ja s	I_NxtJ		;Yes, ignore it and continue scan.
	cmp	al,'0'		;Is byte less than a '0'?
	jb s	I_NxtJ		;Yes, ignore it and continue scan.
	cmp	[di+8].lb,' '	;Have we already found 4 digits?
	jne s	I_CkS0		;Yes, set INVALID & keep scanning.
	push	[di+9].dwd	;Shift "title" bytes 1 place left.
	pop	[di+8].dwd
	mov	[di+11],al	;Insert next "title" message byte.
	and	al,00Fh		;Get cache digit's binary value.
	xchg	ax,[di]		;Multiply current block size by 10.
	mov	dx,10
	mul	dx
	add	[di],ax		;"Add in" next cache-size digit.
	jmp s	I_CkS1		;Go scan more cache-size digits.
I_Term:	mov	ax,04300h	;Inquire if we have an XMS manager.
	call	I_In2F
	mov	dx,(NXMsg-@)	;Point to "No XMS" error message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_EMsg		;No?  Display error message and exit!
	mov	ax,04310h	;Get and save XMS "entry" addresses.
	call	I_In2F
	push	es
	push	bx
	pop	@XEntry.dwd
	mov	di,(LCBlks-@)	;Point to "large cache" block count.
	xor	cx,cx		;Set 5-MB cache size flag.
	cmp	[di].lw,15	;Does user want less than 15-MB?
	jb s	I_CSzF		;Yes, give user a 5-MB cache.
	mov	cl,2		;Set 25-MB cache size flag.
	cmp	[di].lw,25	;Does user want exactly 25-MB?
	je s	I_CSzF		;Yes, set 25-MB cache-size flag.
	dec	cx		;Set 15-MB cache size flag.
	cmp	[di].lw,40	;Does user want less than 40-MB?
	jb s	I_CSzF		;Yes, give user a 15-MB cache.
	mov	cl,4		;Set 50-MB cache size flag.
	cmp	[di].lw,50	;Does user want exactly 50-MB?
	je s	I_CSzF		;Yes, set 50-MB cache-size flag.
	dec	cx		;Set 40-MB cache size flag.
	cmp	[di].lw,80	;Does user want less than 80-MB?
	jb s	I_CSzF		;Yes, give user a 40-MB cache.
	mov	cl,CSDflt	;Set "large" cache size flag.
	cmp	[di].lw,CSMax	;Is cache size invalid or too big?
	ja s	I_CSzE		;Yes?  Ignore & set default cache.
	mov	ax,128		;Set initial 128-MB cache limit.
I_CSz1:	cmp	[di],ax		;User cache size < current limit?
	jb s	I_CSz2		;Yes, set user cache-block count.
	shl	ax,1		;Double current cache-size limit.
	shl	[di+4].dwd,1	;Double variable cache parameters.
	jmp s	I_CSz1		;Go check user's cache size again.
I_CSz2:	shl	[di].lw,4	;Cache blocks = (16 * Megabytes).
	jmp s	I_CSzF		;Go set "large" cache-size flag.
I_CSzE:	mov	[di].lw,(80*16)   ;Error!  Restore default cache.
	mov	[di+8].dwd,"08  "
I_CSzF:	mov	SFlag,cl	  ;Set desired cache-size flag.
	shl	HFlag,1		;Will we be loading in the HMA?
	jz s	I_NoHA		;No, go set upper/DOS memory size.
	mov	ax,04A01h	;Get total "free HMA" space.
	call	I_In2F
	cmp	bx,HMASize	;Enough HMA for our driver logic?
	jae s	I_Siz0		;Yes, check for stand-alone driver.
I_NoHA:	mov	HFlag,0		;Ensure NO use of HMA space!
	mov	ax,HMASize	;Set driver upper/DOS memory size.
	add	VDSLn.lw,ax
I_Siz0:	mov	al,12		;Point to desired cache-size table.
	mul	SFlag
	add	ax,(CachSiz-@)
	xchg	ax,si
	movzx	eax,[si].lw	;Set binary-search limit index.
	mov	@TE1,ax
	mov	ax,[si+4]	;Set binary-search starting offsets.
	mov	@MP1,ax
	mov	@MP2,ax
	mov	ax,[si+2]	;Set cache "granularity" values.
	mov	@GRAN1,al
	mov	@GRAN2,al
	mov	@GRAN3,al
	xchg	ah,@GRSSC
	shr	ax,1		;Multiply number of cache blocks
	mul	[si].lw		;  times (sectors-per-block / 2).
	push	dx		;Get 32-bit cache data XMS size.
	push	ax
	pop	eax
	mov	RqBuf,eax	;Save cache XMS size for below.
	movzx	ecx,[si+6].lw	;Add in cache-tables XMS memory.
	add	eax,ecx
	add	XMSSize,eax	;Save total required XMS memory.
	mov	eax,[si+8]	;Set cache size in "title" message.
	mov	TTL2.dwd,eax
	mov	dx,(TTLMsg-@)	;Display driver "title" message.
	call	I_Msg
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	ax,cs		;Set fixed driver segment values.
	mov	VDSSg.lw,ax
	mov	@CMain.hw,ax
	mov	@DSReg,ax
	mov	@CXSeg,ax
	shl	eax,4		;Set driver's VDS 20-bit address.
	mov	VDSAd,eax
	cli			     ;Avoid interrupts in VDS tests.
	test	es:[VDSFLAG].lb,020h ;Are "VDS services" active?
	jz s	I_REnI		     ;No, re-enable CPU interrupts.
	mov	ax,08103h	;"Lock" driver in memory forever, so
	mov	dx,0000Ch	;  EMM386 may NOT re-map this driver
	call	I_VDS		;  when doing UltraDMA -- bad CRASH!
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
	jc	I_EMsg		;"Lock" error?  Display msg. & exit!
I_REnI:	sti			;Re-enable CPU interrupts.
	mov	eax,VDSAd	;Get final driver 32-bit address.
	add	GDTPAdr,eax	;Relocate "real mode" GDT base addr.
	add	@WrkBf1,eax	;Relocate "working" buffer addresses.
	add	@WrkBf2,eax
	add	@CBAddr,eax	;Relocate cache-entry buffer address.
	mov	dx,CStack.hw	;Get "reserved" XMS memory size.
	or	dx,dx		;Does user want any "reserved" XMS?
	jz s	I_XGet		;No, get driver's actual XMS memory.
	mov	ah,009h		;Get 15-MB or 63-MB "reserved" XMS
	call	I_XMS		;  memory, which we "release" below.
	jnz s	I_XErr		;If error, display message and exit!
	mov	CStack.lw,dx	;Save reserved-XMS "handle" number.
I_XGet:	mov	ah,009h		;Get XMS V2.0 "allocate" command.
	mov	cx,XMSSize.hw	;Do we need 64-MB+ of XMS memory?
	jcxz	I_XReq		;No, request our XMS memory now.
	mov	ah,089h		;Use XMS V3.0 "allocate" command.
I_XReq:	mov	edx,XMSSize	;Request all necessary XMS memory.
	call	I_XMS
	jz s	I_XFre		;If no errors, "free" reserved XMS.
I_XErr:	mov	eax," SMX"	;BAAAD News!  Get "XMS" msg. prefix.
	jmp	I_VErr		;Go display "XMS init error" & exit!
I_XFre:	mov	XMSHndl,dx	;Save our XMS "handle" numbers.
	xchg	CStack.lw,dx
	or	dx,dx		;Any XMS reserved by /R15 or /R63?
	jz s	I_XLok		;No, go "lock" our XMS memory.
	mov	ah,00Ah		;"Free" our reserved XMS memory.
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
I_XLok:	mov	ah,00Ch		;"Lock" our driver's XMS memory.
	mov	dx,CStack.lw
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
	shl	edx,16		;Get unaligned 32-bit buffer address.
	or	dx,bx
	mov	eax,edx		;Copy 32-bit address to EAX-reg.
	jz s	I_XBAd		;Any low-order XMS buffer "offset"?
	mov	ax,0FFFFh	;Yes, align address to an even 64K.
	inc	eax
I_XBAd:	add	@XBase,eax	;Initialize cache-data base address.
	mov	cx,ax		;Get buffer "offset" in XMS memory.
	sub	cx,dx
	jcxz	I_SetB		;Is buffer already on a 64K boundary?
	dec	@XBase.hw	;No, decrement base address by 64K.
I_SetB:	mov	eax,RqBuf	;Get needed cache XMS in 1K blocks.
	shl	eax,10		;Convert from 1K blocks to bytes.
	add	eax,@XBase	;Add cache-data XMS base address.
	mov	@CTBas1,eax	;Set XMS cache-table base addresses.
	add	@CTBas2,eax
	add	@CTBas3,eax
	xchg	eax,edx		;Save cache-table base in EDX-reg.
	movzx	eax,@TE1	;Get binary-search table size.
	shl	eax,1
	mov	ecx,eax		;Save search-table size in ECX-reg.
	shl	eax,1		;Get offset to binary-search table.
	add	eax,ecx
	shl	eax,1
	add	eax,edx		;Set binary-search table addresses.
	mov	@STAdr1,eax
	mov	@STAdr2,eax
	mov	@STAdr3,eax
	add	eax,ecx		;Set search-table buffer address.
	mov	@SBuff,eax
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
	jnz	I_A20E		;If any "A20" error, bail out NOW!
	mov	dx,@TE1		;Initialize search-table count.
	xor	bp,bp		;Initialize search-table index.
I_RST1:	mov	ax,bp		;Set next 50 search-table indexes
	xor	ecx,ecx		;  in init tables & messages area.
	mov	cl,50
	mov	si,(CachSiz-@)
I_RST2:	mov	[si],ax
	inc	ax
	inc	si
	inc	si
	loop	I_RST2
	xor	esi,esi		;Set 32-bit move source address.
	mov	si,(CachSiz-@)	;(Offset of our indexes buffer +
	add	esi,VDSAd	;  32-bit driver "base" address).
	movzx	edi,bp		;Set 32-bit move destination addr.
	shl	edi,1		;(2 * current "reset" index +
	add	edi,@STAdr1	;  binary-search table address).
	mov	bp,ax		;Update next cache-table index.
	mov	cl,50		;Get 50-word move length.
	cmp	cx,dx		;At least 50 words left to go?
	jbe s	I_RST3		;Yes, use full 50-word count.
	mov	cx,dx		;Use remaining word count.
I_RST3:	shl	cx,1		;Convert word count to byte count.
	push	dx		;Save move count and cache index.
	push	bp
	call	MvData		;Move 50 indexes into search table.
	pop	bp		;Reload cache index and move count.
	pop	dx
	jnc s	I_RST4		;If no XMS errors, check move count.
	call	I_A20D		;BAD News!  Do "A20 local-disable".
	jmp	I_XErr		;Go display "XMS init error" & exit!
I_RST4:	sub	dx,50		;More search-table indexes to set?
	ja s	I_RST1		;Yes, loop back and do next 50.
	call	I_A20D		;Issue "A20 local-disable" request.
	shr	HFlag,1		;Will we be loading in the HMA?
	jz	I_Done		;No, go post "Init" packet results.
	mov	ax,04A02h	;Request needed memory in the HMA.
	mov	bx,HMASize
	call	I_In2F
	push	es		;Get 32-bit HMA segment/offset addr.
	push	di
	pop	eax
	mov	@CMain,eax	;Set cache entry-routine pointer.
	inc	eax		;Is our HMA address = -1 (no HMA)?
	jnz s	I_HMA1		;No, do all needed HMA adjustments.
	mov	eax," AMH"	;Get "HMA" error-message prefix.
I_HMAX:	jmp	I_VErr		;Go display error message & exit!
I_HMA1:	xor	ecx,ecx		;Get 32-bit HMA segment address.
	mov	cx,es
	shl	ecx,4
	movzx	eax,di		;Adjust real-mode GDT base addr.
	add	ax,(GDT-Cache)
	add	eax,ecx
	mov	GDTPAdr,eax
	lea	ax,[di-(Cache-@)] ;Get caching HMA logic offset.
	add	@GDTP,ax	;Adjust "MvData" routine offsets.
	add	@MvDesc,ax
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
I_A20E:	mov	eax," 02A"	;Get "A20" error-message prefix.
	jnz s	I_HMAX		;If any "A20" error, bail out QUICK!
	mov     cx,HMASize      ;Move required logic up to the HMA.
	mov	si,(Cache-@)
	les	di,@CMain
	rep	movsb
	call	I_A20D		;Issue "A20 local-disable" request.
I_Done:	xor	ax,ax		;Done!  Load & reset driver length.
	xchg	ax,VDSLn.lw
	les	bx,InitPkt	;Set results in DOS "Init" packet.
	mov	es:[bx].RPLen,ax
	mov	es:[bx].RPStat,RPDON
	mov	@Stack,ax	;Set initial driver stack pointer.
	xor	bx,bx		;Clear driver stack (helps debug).
	xchg	ax,bx
	mov	cx,STACK
I_ClrS:	dec	bx
	mov	[bx],al
	loop	I_ClrS
	popad			;Reload all 32-bit CPU registers.
I_Bye:	pop	dx		;Reload 16-bit registers we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; Init subroutine to convert 4-digit hex values to ASCII for display.
;   At entry, the value is in the AX-reg., and the message pointer is
;   in the SI-reg.   The SI-reg. is updated.   The CX-reg. is zeroed.
;
I_Hex:	mov	cx,4		;Set 4-digit count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Store next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
	ret			;Exit.
;
; Subroutines to issue initialization "external" calls.
;
I_A20D:	mov	ah,006h		;"A20 local-disable" -- get XMS code.
I_XMS:	call	@XEntry.dwd	;XMS -- issue desired request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_VDS:	mov	di,(VDSLn-@)	;VDS -- Point to parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
	jmp s	I_IntX		;Restore driver settings, then exit.
I_Msg:	push	bx		;Message -- save our BX-register.
	mov	ah,009h		;Issue DOS "display string" request.
	int	021h
	pop	bx		;Reload our BX-register.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In21:	int	021h		;General DOS request -- issue Int 21h.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In2F:	int	02Fh		;"Multiplex" -- issue XMS/HMA request.
I_IntX:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
I_Ret:	ret			;Exit.
CODE	ends
	end
